/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.vpnconnection;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.vpnconnection.model.CreateVpnConnectionRequest;
import com.nifty.cloud.sdk.vpnconnection.model.CreateVpnConnectionResult;
import com.nifty.cloud.sdk.vpnconnection.model.DeleteVpnConnectionRequest;
import com.nifty.cloud.sdk.vpnconnection.model.DeleteVpnConnectionResult;
import com.nifty.cloud.sdk.vpnconnection.model.DescribeVpnConnectionsRequest;
import com.nifty.cloud.sdk.vpnconnection.model.DescribeVpnConnectionsResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * VPNコネクションカテゴリクライアント。<br />
 * VPNコネクション カテゴリの非同期クライアントの具象クラスです。<br />
 * 戻り値の"利用不可"パラメータは返却値がnullまたは空となります。
 */
public class NiftyVpnConnectionAsyncClient extends NiftyVpnConnectionClient implements NiftyVpnConnectionAsync {


	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、VPNコネクションカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftyVpnConnectionAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報、クライアント設定を指定し、VPNコネクションカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftyVpnConnectionAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、VPNコネクションカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyVpnConnectionAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;

	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、VPNコネクションカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyVpnConnectionAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.vpnconnection.NiftyVpnConnectionAsync#createVpnConnectionAsync(com.nifty.cloud.sdk.vpnconnection.model.CreateVpnConnectionRequest)
	 */
	@Override
	public Future<CreateVpnConnectionResult> createVpnConnectionAsync(final CreateVpnConnectionRequest request) {
		return executorService.submit(
				new Callable<CreateVpnConnectionResult>() {
					@Override
					public CreateVpnConnectionResult call() {
						return createVpnConnection(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.vpnconnection.NiftyVpnConnectionAsync#deleteVpnConnectionAsync(com.nifty.cloud.sdk.vpnconnection.model.DeleteVpnConnectionRequest)
	 */
	@Override
	public Future<DeleteVpnConnectionResult> deleteVpnConnectionAsync(final DeleteVpnConnectionRequest request) {
		return executorService.submit(
				new Callable<DeleteVpnConnectionResult>() {
					@Override
					public DeleteVpnConnectionResult call() {
						return deleteVpnConnection(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.vpnconnection.NiftyVpnConnectionAsync#describeVpnConnectionsAsync(com.nifty.cloud.sdk.vpnconnection.model.DescribeVpnConnectionsRequest)
	 */
	@Override
	public Future<DescribeVpnConnectionsResult> describeVpnConnectionsAsync(final DescribeVpnConnectionsRequest request) {
		return executorService.submit(
				new Callable<DescribeVpnConnectionsResult>() {
					@Override
					public DescribeVpnConnectionsResult call() {
						return describeVpnConnections(request);
					}
				}
		);
	}
}