/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.usage.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.usage.model.transform.UsageDetailSetConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * VPNゲートウェイ利用情報クラス。<br />
 * このクラスはVPNゲートウェイ利用情報を格納します。
 */
public class UsageVpnGateway {

	/** VPNゲートウェイ月額利用情報 */
	@XStreamAlias("vpnGatewayMonthlyRateSet")
	@XStreamConverter(UsageDetailSetConverter.class)
	private List<UsageDetail> vpnGatewayMonthlyRates;

	/** VPNゲートウェイ従量利用情報 */
	@XStreamAlias("vpnGatewayMeasuredRateSet")
	@XStreamConverter(UsageDetailSetConverter.class)
	private List<UsageDetail> vpnGatewayMeasuredRates;

	/**
	 * VPNゲートウェイ月額利用情報を取得します。
	 *
	 * @return VPNゲートウェイ月額利用情報
	 */
	public List<UsageDetail> getVpnGatewayMonthlyRates() {
		return vpnGatewayMonthlyRates;
	}

	/**
	 * VPNゲートウェイ月額利用情報を設定します。
	 *
	 * @param vpnGatewayMonthlyRates VPNゲートウェイ月額利用情報
	 */
	public void setVpnGatewayMonthlyRates(List<UsageDetail> vpnGatewayMonthlyRates) {
		this.vpnGatewayMonthlyRates = vpnGatewayMonthlyRates;
	}

	/**
	 * VPNゲートウェイ月額利用情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param vpnGatewayMonthlyRates VPNゲートウェイ月額利用情報の配列
	 * @return 自オブジェクト
	 */
	public UsageVpnGateway withVpnGatewayMonthlyRates(UsageDetail ... vpnGatewayMonthlyRates) {
		if (this.vpnGatewayMonthlyRates == null) this.vpnGatewayMonthlyRates = new ArrayList<UsageDetail>();
		for (UsageDetail vpnGatewayMonthlyRate : vpnGatewayMonthlyRates) {
			getVpnGatewayMonthlyRates().add(vpnGatewayMonthlyRate);
		}
		return this;
	}

	/**
	 * VPNゲートウェイ月額利用情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param vpnGatewayMonthlyRates VPNゲートウェイ月額利用情報リスト
	 * @return 自オブジェクト
	 */
	public UsageVpnGateway withVpnGatewayMonthlyRates(Collection<UsageDetail> vpnGatewayMonthlyRates) {
		if (this.vpnGatewayMonthlyRates == null) this.vpnGatewayMonthlyRates = new ArrayList<UsageDetail>();
		if (vpnGatewayMonthlyRates != null) {
			getVpnGatewayMonthlyRates().addAll(vpnGatewayMonthlyRates);
		}
		return this;
	}

	/**
	 * VPNゲートウェイ従量利用情報を取得します。
	 *
	 * @return VPNゲートウェイ従量利用情報
	 */
	public List<UsageDetail> getVpnGatewayMeasuredRates() {
		return vpnGatewayMeasuredRates;
	}

	/**
	 * VPNゲートウェイ従量利用情報を設定します。
	 *
	 * @param vpnGatewayMeasuredRates VPNゲートウェイ従量利用情報
	 */
	public void setVpnGatewayMeasuredRates(List<UsageDetail> vpnGatewayMeasuredRates) {
		this.vpnGatewayMeasuredRates = vpnGatewayMeasuredRates;
	}

	/**
	 * VPNゲートウェイ従量利用情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param vpnGatewayMeasuredRates VPNゲートウェイ従量利用情報の配列
	 * @return 自オブジェクト
	 */
	public UsageVpnGateway withVpnGatewayMeasuredRates(UsageDetail ... vpnGatewayMeasuredRates) {
		if (this.vpnGatewayMeasuredRates == null) this.vpnGatewayMeasuredRates = new ArrayList<UsageDetail>();
		for (UsageDetail vpnGatewayMeasuredRate : vpnGatewayMeasuredRates) {
			getVpnGatewayMeasuredRates().add(vpnGatewayMeasuredRate);
		}
		return this;
	}

	/**
	 * VPNゲートウェイ従量利用情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param vpnGatewayMeasuredRates VPNゲートウェイ従量利用情報リスト
	 * @return 自オブジェクト
	 */
	public UsageVpnGateway withVpnGatewayMeasuredRates(Collection<UsageDetail> vpnGatewayMeasuredRates) {
		if (this.vpnGatewayMeasuredRates == null) this.vpnGatewayMeasuredRates = new ArrayList<UsageDetail>();
		if (vpnGatewayMeasuredRates != null) {
			getVpnGatewayMeasuredRates().addAll(vpnGatewayMeasuredRates);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[vpnGatewayMonthlyRates=");
		builder.append(vpnGatewayMonthlyRates);
		builder.append(", vpnGatewayMeasuredRates=");
		builder.append(vpnGatewayMeasuredRates);
		builder.append("]");
		return builder.toString();
	}
}
