/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.usage.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.usage.model.transform.UsageDetailSetConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * セキュアネットワーク利用情報クラス。<br />
 * このクラスはセキュアネットワーク利用情報を格納します。
 */
public class UsageSecureNet {

	/** セキュアネットワーク基本利用情報 */
	@XStreamAlias("secureNetMonthlyRate")
	private UsageDetail secureNetMonthlyRate;

	/** VPN接続利用情報 */
	@XStreamAlias("vpnConnectMeasuredRateSet")
	@XStreamConverter(UsageDetailSetConverter.class)
	private List<UsageDetail> vpnConnectMeasuredRates;

	/**
	 * セキュアネットワーク基本利用情報を取得します。
	 *
	 * @return セキュアネットワーク基本利用情報
	 */
	public UsageDetail getSecureNetMonthlyRate() {
		return secureNetMonthlyRate;
	}

	/**
	 * セキュアネットワーク基本利用情報を設定します。
	 *
	 * @param secureNetMonthlyRate セキュアネットワーク基本利用情報
	 */
	public void setSecureNetMonthlyRate(UsageDetail secureNetMonthlyRate) {
		this.secureNetMonthlyRate = secureNetMonthlyRate;
	}

	/**
	 * セキュアネットワーク基本利用情報を設定し、自オブジェクトを返します。
	 *
	 * @param secureNetMonthlyRate セキュアネットワーク基本利用情報
	 * @return 自オブジェクト
	 */
	public UsageSecureNet withSecureNetMonthlyRate(UsageDetail secureNetMonthlyRate) {
		setSecureNetMonthlyRate(secureNetMonthlyRate);
		return this;
	}

	/**
	 * VPN接続利用情報を取得します。
	 *
	 * @return VPN接続利用情報
	 */
	public List<UsageDetail> getVpnConnectMeasuredRates() {
		return vpnConnectMeasuredRates;
	}

	/**
	 * VPN接続利用情報を設定します。
	 *
	 * @param vpnConnectMeasuredRates VPN接続利用情報
	 */
	public void setVpnConnectMeasuredRates(List<UsageDetail> vpnConnectMeasuredRates) {
		this.vpnConnectMeasuredRates = vpnConnectMeasuredRates;
	}

	/**
	 * VPN接続利用情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param vpnConnectMeasuredRates VPN接続利用情報の配列
	 * @return 自オブジェクト
	 */
	public UsageSecureNet withVpnConnectMeasuredRates(UsageDetail ... vpnConnectMeasuredRates) {
		if (this.vpnConnectMeasuredRates == null) this.vpnConnectMeasuredRates = new ArrayList<UsageDetail>();
		for (UsageDetail vpnConnectMeasuredRate : vpnConnectMeasuredRates) {
			getVpnConnectMeasuredRates().add(vpnConnectMeasuredRate);
		}
		return this;
	}

	/**
	 * VPN接続利用情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param vpnConnectMeasuredRates VPN接続利用情報リスト
	 * @return 自オブジェクト
	 */
	public UsageSecureNet withVpnConnectMeasuredRates(Collection<UsageDetail> vpnConnectMeasuredRates) {
		if (this.vpnConnectMeasuredRates == null) this.vpnConnectMeasuredRates = new ArrayList<UsageDetail>();
		if (vpnConnectMeasuredRates != null) {
			getVpnConnectMeasuredRates().addAll(vpnConnectMeasuredRates);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[secureNetMonthlyRate=");
		builder.append(secureNetMonthlyRate);
		builder.append(", vpnConnectMeasuredRates=");
		builder.append(vpnConnectMeasuredRates);
		builder.append("]");
		return builder.toString();
	}
}
