/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.usage.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.usage.model.transform.UsageDetailSetConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * ルーター利用情報クラス。<br />
 * このクラスはルーター利用情報を格納します。
 */
public class UsageRouter {

	/** ルーター月額利用情報 */
	@XStreamAlias("routerMonthlyRateSet")
	@XStreamConverter(UsageDetailSetConverter.class)
	private List<UsageDetail> routerMonthlyRates;

	/** ルーター従量利用情報 */
	@XStreamAlias("routerMeasuredRateSet")
	@XStreamConverter(UsageDetailSetConverter.class)
	private List<UsageDetail> routerMeasuredRates;

	/**
	 * ルーター月額利用情報を取得します。
	 *
	 * @return ルーター月額利用情報
	 */
	public List<UsageDetail> getRouterMonthlyRates() {
		return routerMonthlyRates;
	}

	/**
	 * ルーター月額利用情報を設定します。
	 *
	 * @param routerMonthlyRates ルーター月額利用情報
	 */
	public void setRouterMonthlyRates(List<UsageDetail> routerMonthlyRates) {
		this.routerMonthlyRates = routerMonthlyRates;
	}

	/**
	 * ルーター月額利用情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param routerMonthlyRates ルーター月額利用情報の配列
	 * @return 自オブジェクト
	 */
	public UsageRouter withRouterMonthlyRates(UsageDetail ... routerMonthlyRates) {
		if (this.routerMonthlyRates == null) this.routerMonthlyRates = new ArrayList<UsageDetail>();
		for (UsageDetail routerMonthlyRate : routerMonthlyRates) {
			getRouterMonthlyRates().add(routerMonthlyRate);
		}
		return this;
	}

	/**
	 * ルーター月額利用情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param routerMonthlyRates ルーター月額利用情報リスト
	 * @return 自オブジェクト
	 */
	public UsageRouter withRouterMonthlyRates(Collection<UsageDetail> routerMonthlyRates) {
		if (this.routerMonthlyRates == null) this.routerMonthlyRates = new ArrayList<UsageDetail>();
		if (routerMonthlyRates != null) {
			getRouterMonthlyRates().addAll(routerMonthlyRates);
		}
		return this;
	}

	/**
	 * ルーター従量利用情報を取得します。
	 *
	 * @return ルーター従量利用情報
	 */
	public List<UsageDetail> getRouterMeasuredRates() {
		return routerMeasuredRates;
	}

	/**
	 * ルーター従量利用情報を設定します。
	 *
	 * @param routerMeasuredRates ルーター従量利用情報
	 */
	public void setRouterMeasuredRates(List<UsageDetail> routerMeasuredRates) {
		this.routerMeasuredRates = routerMeasuredRates;
	}

	/**
	 * ルーター従量利用情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param routerMeasuredRates ルーター従量利用情報の配列
	 * @return 自オブジェクト
	 */
	public UsageRouter withRouterMeasuredRates(UsageDetail ... routerMeasuredRates) {
		if (this.routerMeasuredRates == null) this.routerMeasuredRates = new ArrayList<UsageDetail>();
		for (UsageDetail routerMeasuredRate : routerMeasuredRates) {
			getRouterMeasuredRates().add(routerMeasuredRate);
		}
		return this;
	}

	/**
	 * ルーター従量利用情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param routerMeasuredRates ルーター従量利用情報リスト
	 * @return 自オブジェクト
	 */
	public UsageRouter withRouterMeasuredRates(Collection<UsageDetail> routerMeasuredRates) {
		if (this.routerMeasuredRates == null) this.routerMeasuredRates = new ArrayList<UsageDetail>();
		if (routerMeasuredRates != null) {
			getRouterMeasuredRates().addAll(routerMeasuredRates);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[routerMonthlyRates=");
		builder.append(routerMonthlyRates);
		builder.append(", routerMeasuredRates=");
		builder.append(routerMeasuredRates);
		builder.append("]");
		return builder.toString();
	}
}
