/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.usage.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.usage.model.transform.UsageDetailSetConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * インターネットVPN利用情報クラス。<br />
 * このクラスはインターネットVPN利用情報を格納します。
 */
public class UsageInternetVpn {

	/** インターネットVPN初期利用情報 */
	@XStreamAlias("internetVpnInitial")
	private UsageDetail internetVpnInitial;

	/** インターネットVPN月額利用情報 */
	@XStreamAlias("internetVpnMonthlyRateSet")
	@XStreamConverter(UsageDetailSetConverter.class)
	private List<UsageDetail> internetVpnMonthlyRates;

	/**
	 * インターネットVPN初期利用情報を取得します。
	 *
	 * @return インターネットVPN初期利用情報
	 */
	public UsageDetail getInternetVpnInitial() {
		return internetVpnInitial;
	}

	/**
	 * インターネットVPN初期利用情報を設定します。
	 *
	 * @param internetVpnInitial インターネットVPN初期利用情報
	 */
	public void setInternetVpnInitial(UsageDetail internetVpnInitial) {
		this.internetVpnInitial = internetVpnInitial;
	}

	/**
	 * インターネットVPN初期利用情報を設定し、自オブジェクトを返します。
	 *
	 * @param internetVpnInitial インターネットVPN初期利用情報
	 * @return 自オブジェクト
	 */
	public UsageInternetVpn withInternetVpnInitial(UsageDetail internetVpnInitial) {
		setInternetVpnInitial(internetVpnInitial);
		return this;
	}

	/**
	 * インターネットVPN月額利用情報を取得します。
	 *
	 * @return インターネットVPN月額利用情報
	 */
	public List<UsageDetail> getInternetVpnMonthlyRates() {
		return internetVpnMonthlyRates;
	}

	/**
	 * インターネットVPN月額利用情報を設定します。
	 *
	 * @param internetVpnMonthlyRates インターネットVPN月額利用情報
	 */
	public void setInternetVpnMonthlyRates(List<UsageDetail> internetVpnMonthlyRates) {
		this.internetVpnMonthlyRates = internetVpnMonthlyRates;
	}

	/**
	 * インターネットVPN月額利用情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param internetVpnMonthlyRates インターネットVPN月額利用情報の配列
	 * @return 自オブジェクト
	 */
	public UsageInternetVpn withInternetVpnMonthlyRates(UsageDetail ... internetVpnMonthlyRates) {
		if (this.internetVpnMonthlyRates == null) this.internetVpnMonthlyRates = new ArrayList<UsageDetail>();
		for (UsageDetail internetVpnMonthlyRate : internetVpnMonthlyRates) {
			getInternetVpnMonthlyRates().add(internetVpnMonthlyRate);
		}
		return this;
	}

	/**
	 * インターネットVPN月額利用情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param internetVpnMonthlyRates インターネットVPN月額利用情報リスト
	 * @return 自オブジェクト
	 */
	public UsageInternetVpn withInternetVpnMonthlyRates(Collection<UsageDetail> internetVpnMonthlyRates) {
		if (this.internetVpnMonthlyRates == null) this.internetVpnMonthlyRates = new ArrayList<UsageDetail>();
		if (internetVpnMonthlyRates != null) {
			getInternetVpnMonthlyRates().addAll(internetVpnMonthlyRates);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[internetVpnInitial=");
		builder.append(internetVpnInitial);
		builder.append(", internetVpnMonthlyRates=");
		builder.append(internetVpnMonthlyRates);
		builder.append("]");
		return builder.toString();
	}
}
