/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.usage.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.usage.model.transform.UsageDetailSetConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * マルチロードバランサー利用情報クラス。<br />
 * このクラスはマルチロードバランサー利用情報を格納します。
 */
public class ElasticLoadBalancer {

    /** マルチロードバランサー月額利用VIP情報リスト */
    @XStreamAlias("vip")
    @XStreamConverter(UsageDetailSetConverter.class)
    private List<UsageDetail> vips;

    /** マルチロードバランサー従量利用VIP情報リスト */
    @XStreamAlias("vipMeasuredRate")
    @XStreamConverter(UsageDetailSetConverter.class)
    private List<UsageDetail> vipMeasuredRates;

    /**
     * マルチロードバランサー月額利用VIP情報リストを取得します。
     *
     * @return マルチロードバランサー月額利用VIP情報リスト
     */
    public List<UsageDetail> getVips() {
        return vips;
    }

    /**
     * マルチロードバランサー月額利用VIP情報リストを設定します。
     *
     * @param vips
     *            マルチロードバランサー月額利用VIP情報リスト
     */
    public void setVips(List<UsageDetail> vips) {
        this.vips = vips;
    }

    /**
     * マルチロードバランサー月額利用VIP情報の配列を設定し、自オブジェクトを返します。
     *
     * @param vips
     *            マルチロードバランサー月額利用VIP情報の配列
     * @return 自オブジェクト
     */
    public ElasticLoadBalancer withVips(UsageDetail... vips) {
        if (this.vips == null)
            this.vips = new ArrayList<UsageDetail>();
        for (UsageDetail vip : vips) {
            getVips().add(vip);
        }
        return this;
    }

    /**
     * マルチロードバランサー月額利用VIP情報リストを設定し、自オブジェクトを返します。
     *
     * @param vips
     *            マルチロードバランサー月額利用VIP情報リスト
     * @return 自オブジェクト
     */
    public ElasticLoadBalancer withVips(Collection<UsageDetail> vips) {
        if (this.vips == null)
            this.vips = new ArrayList<UsageDetail>();
        if (vips != null) {
            getVips().addAll(vips);
        }
        return this;
    }

    /**
     * マルチロードバランサー従量利用VIP情報リストを取得します。
     *
     * @return マルチロードバランサー従量利用VIP情報リスト
     */
    public List<UsageDetail> getVipMeasuredRates() {
        return vipMeasuredRates;
    }

    /**
     * マルチロードバランサー従量利用VIP情報リストを設定します。
     *
     * @param vipMeasuredRates
     *            マルチロードバランサー従量利用VIP情報リスト
     */
    public void setVipMeasuredRates(List<UsageDetail> vipMeasuredRates) {
        this.vipMeasuredRates = vipMeasuredRates;
    }

    /**
     * マルチロードバランサー従量利用VIP情報の配列を設定し、自オブジェクトを返します。
     *
     * @param vipMeasuredRates
     *            マルチロードバランサー従量利用VIP情報の配列
     * @return 自オブジェクト
     */
    public ElasticLoadBalancer withVpnGatewayMeasuredRates(UsageDetail... vipMeasuredRates) {
        if (this.vipMeasuredRates == null)
            this.vipMeasuredRates = new ArrayList<UsageDetail>();
        for (UsageDetail vipMeasuredRate : vipMeasuredRates) {
            getVipMeasuredRates().add(vipMeasuredRate);
        }
        return this;
    }

    /**
     * マルチロードバランサー従量利用VIP情報リストを設定し、自オブジェクトを返します。
     *
     * @param vipMeasuredRates
     *            マルチロードバランサー従量利用VIP情報リスト
     * @return 自オブジェクト
     */
    public ElasticLoadBalancer
        withVpnGatewayMeasuredRates(Collection<UsageDetail> vipMeasuredRates) {
        if (this.vipMeasuredRates == null)
            this.vipMeasuredRates = new ArrayList<UsageDetail>();
        if (vipMeasuredRates != null) {
            getVipMeasuredRates().addAll(vipMeasuredRates);
        }
        return this;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("[vips=");
        builder.append(vips);
        builder.append(", vipMeasuredRates=");
        builder.append(vipMeasuredRates);
        builder.append("]");
        return builder.toString();
    }
}
