/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.snapshot.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Result;
import com.nifty.cloud.sdk.snapshot.model.transform.NiftyCreateInstanceSnapshotResultConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * NiftyCreateInstanceSnapshot処理結果クラス。<br />
 * このクラスはNiftyCreateInstanceSnapshotのレスポンスを格納します。
 */
@XStreamAlias("NiftyCreateInstanceSnapshotResponse")
@XStreamConverter(NiftyCreateInstanceSnapshotResultConverter.class)
public class NiftyCreateInstanceSnapshotResult extends Result {

	/** リクエスト識別子 */
	private String requestId;

	/** スナップショット名 */
	private String snapshotName;

	/** サーバー設定情報 */
	private List<SnapshotInstance> instances;

	/**
	 * リクエスト識別子を取得します。
	 *
	 * @return リクエスト識別子
	 */
	public String getRequestId() {
		return requestId;
	}

	/**
	 * リクエスト識別子を設定します。
	 *
	 * @param requestId リクエスト識別子
	 */
	public void setRequestId(String requestId) {
		this.requestId = requestId;
	}

	/**
	 * リクエスト識別子を設定し、自オブジェクトを返します。
	 *
	 * @param requestId リクエスト識別子
	 * @return 自オブジェクト
	 */
	public NiftyCreateInstanceSnapshotResult withRequestId(String requestId) {
		setRequestId(requestId);
		return this;
	}

	/**
	 * スナップショット名を取得します。
	 *
	 * @return スナップショット名
	 */
	public String getSnapshotName() {
		return snapshotName;
	}

	/**
	 * スナップショット名を設定します。
	 *
	 * @param snapshotName スナップショット名
	 */
	public void setSnapshotName(String snapshotName) {
		this.snapshotName = snapshotName;
	}

	/**
	 * スナップショット名を設定し、自オブジェクトを返します。
	 *
	 * @param snapshotName スナップショット名
	 * @return 自オブジェクト
	 */
	public NiftyCreateInstanceSnapshotResult withSnapshotName(String snapshotName) {
		setSnapshotName(snapshotName);
		return this;
	}

	/**
	 * サーバー設定情報を取得します。
	 *
	 * @return サーバー設定情報
	 */
	public List<SnapshotInstance> getInstances() {
		return instances;
	}

	/**
	 * サーバー設定情報を設定します。
	 *
	 * @param instances サーバー設定情報
	 */
	public void setInstances(List<SnapshotInstance> instances) {
		this.instances = instances;
	}

	/**
	 * サーバー設定情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param instances サーバー設定情報の配列
	 * @return 自オブジェクト
	 */
	public NiftyCreateInstanceSnapshotResult withInstances(SnapshotInstance ... instances) {
		if (this.instances == null) this.instances = new ArrayList<SnapshotInstance>();
		for (SnapshotInstance instance : instances) {
			getInstances().add(instance);
		}
		return this;
	}

	/**
	 * サーバー設定情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param instances サーバー設定情報リスト
	 * @return 自オブジェクト
	 */
	public NiftyCreateInstanceSnapshotResult withInstances(Collection<SnapshotInstance> instances) {
		if (this.instances == null) this.instances = new ArrayList<SnapshotInstance>();
		if (instances != null) {
			getInstances().addAll(instances);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[requestId=");
		builder.append(requestId);
		builder.append(", snapshotName=");
		builder.append(snapshotName);
		builder.append(", instances=");
		builder.append(instances);
		builder.append("]");
		return builder.toString();
	}
}
