/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.snapshot;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.snapshot.model.NiftyCreateInstanceSnapshotRequest;
import com.nifty.cloud.sdk.snapshot.model.NiftyCreateInstanceSnapshotResult;
import com.nifty.cloud.sdk.snapshot.model.NiftyDeleteInstanceSnapshotRequest;
import com.nifty.cloud.sdk.snapshot.model.NiftyDeleteInstanceSnapshotResult;
import com.nifty.cloud.sdk.snapshot.model.NiftyDescribeInstanceSnapshotsRequest;
import com.nifty.cloud.sdk.snapshot.model.NiftyDescribeInstanceSnapshotsResult;
import com.nifty.cloud.sdk.snapshot.model.NiftyModifyInstanceSnapshotAttributeRequest;
import com.nifty.cloud.sdk.snapshot.model.NiftyModifyInstanceSnapshotAttributeResult;
import com.nifty.cloud.sdk.snapshot.model.NiftyRestoreInstanceSnapshotRequest;
import com.nifty.cloud.sdk.snapshot.model.NiftyRestoreInstanceSnapshotResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * スナップショットカテゴリクライアント。<br />
 * スナップショット カテゴリの非同期クライアントの具象クラスです。
 *
 */
public class NiftySnapshotAsyncClient extends NiftySnapshotClient implements NiftySnapshotAsync {


	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、スナップショットカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftySnapshotAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報、クライアント設定を指定し、スナップショットカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftySnapshotAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、スナップショットカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftySnapshotAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;

	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、スナップショットカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftySnapshotAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.snapshot.NiftySnapshotAsync#niftyCreateInstanceSnapshotAsync(com.nifty.cloud.sdk.snapshot.model.NiftyCreateInstanceSnapshotRequest)
	 */
	@Override
	public Future<NiftyCreateInstanceSnapshotResult> niftyCreateInstanceSnapshotAsync(final NiftyCreateInstanceSnapshotRequest request) {
		return executorService.submit(
				new Callable<NiftyCreateInstanceSnapshotResult>() {
					@Override
					public NiftyCreateInstanceSnapshotResult call() {
						return niftyCreateInstanceSnapshot(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.snapshot.NiftySnapshotAsync#niftyModifyInstanceSnapshotAttributeAsync(com.nifty.cloud.sdk.snapshot.model.NiftyModifyInstanceSnapshotAttributeRequest)
	 */
	@Override
	public Future<NiftyModifyInstanceSnapshotAttributeResult> niftyModifyInstanceSnapshotAttributeAsync(final NiftyModifyInstanceSnapshotAttributeRequest request) {
		return executorService.submit(
				new Callable<NiftyModifyInstanceSnapshotAttributeResult>() {
					@Override
					public NiftyModifyInstanceSnapshotAttributeResult call() {
						return niftyModifyInstanceSnapshotAttribute(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.snapshot.NiftySnapshotAsync#niftyDescribeInstanceSnapshotsAsync(com.nifty.cloud.sdk.snapshot.model.NiftyDescribeInstanceSnapshotsRequest)
	 */
	@Override
	public Future<NiftyDescribeInstanceSnapshotsResult> niftyDescribeInstanceSnapshotsAsync(final NiftyDescribeInstanceSnapshotsRequest request) {
		return executorService.submit(
				new Callable<NiftyDescribeInstanceSnapshotsResult>() {
					@Override
					public NiftyDescribeInstanceSnapshotsResult call() {
						return niftyDescribeInstanceSnapshots(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.snapshot.NiftySnapshotAsync#niftyDeleteInstanceSnapshotAsync(com.nifty.cloud.sdk.snapshot.model.NiftyDeleteInstanceSnapshotRequest)
	 */
	@Override
	public Future<NiftyDeleteInstanceSnapshotResult> niftyDeleteInstanceSnapshotAsync(final NiftyDeleteInstanceSnapshotRequest request) {
		return executorService.submit(
				new Callable<NiftyDeleteInstanceSnapshotResult>() {
					@Override
					public NiftyDeleteInstanceSnapshotResult call() {
						return niftyDeleteInstanceSnapshot(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.snapshot.NiftySnapshotAsync#niftyRestoreInstanceSnapshotAsync(com.nifty.cloud.sdk.snapshot.model.NiftyRestoreInstanceSnapshotRequest)
	 */
	@Override
	public Future<NiftyRestoreInstanceSnapshotResult> niftyRestoreInstanceSnapshotAsync(final NiftyRestoreInstanceSnapshotRequest request) {
		return executorService.submit(
				new Callable<NiftyRestoreInstanceSnapshotResult>() {
					@Override
					public NiftyRestoreInstanceSnapshotResult call() {
						return niftyRestoreInstanceSnapshot(request);
					}
				}
		);
	}
}