/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.separate;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.separate.model.NiftyCreateSeparateInstanceRuleRequest;
import com.nifty.cloud.sdk.separate.model.NiftyCreateSeparateInstanceRuleResult;
import com.nifty.cloud.sdk.separate.model.NiftyDeleteSeparateInstanceRuleRequest;
import com.nifty.cloud.sdk.separate.model.NiftyDeleteSeparateInstanceRuleResult;
import com.nifty.cloud.sdk.separate.model.NiftyDeregisterInstancesFromSeparateInstanceRuleRequest;
import com.nifty.cloud.sdk.separate.model.NiftyDeregisterInstancesFromSeparateInstanceRuleResult;
import com.nifty.cloud.sdk.separate.model.NiftyDescribeSeparateInstanceRulesRequest;
import com.nifty.cloud.sdk.separate.model.NiftyDescribeSeparateInstanceRulesResult;
import com.nifty.cloud.sdk.separate.model.NiftyRegisterInstancesWithSeparateInstanceRuleRequest;
import com.nifty.cloud.sdk.separate.model.NiftyRegisterInstancesWithSeparateInstanceRuleResult;
import com.nifty.cloud.sdk.separate.model.NiftyUpdateSeparateInstanceRuleRequest;
import com.nifty.cloud.sdk.separate.model.NiftyUpdateSeparateInstanceRuleResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。 サーバーセパレートカテゴリクライアント。<br />
 * サーバーセパレートカテゴリの非同期クライアントの具象クラスです。<br />
 * 戻り値の"利用不可"パラメータは返却値がnullまたは空となります。
 *
 */
public class NiftySeparateAsyncClient extends NiftySeparateClient
    implements NiftySeparateAsync {

    /** 非同期タスク実行クラス. */
    protected ExecutorService executorService = null;

    /**
     * 認証情報を指定し、サーバーカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     */
    public NiftySeparateAsyncClient(Credentials credential) {
        this(credential, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報、クライアント設定を指定し、サーバーカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     */
    public NiftySeparateAsyncClient(Credentials credential,
        ClientConfiguration configuration) {
        this(credential, configuration, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報,非同期タスク実行クラスを指定し、サーバーカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftySeparateAsyncClient(Credentials credential, ExecutorService executorService) {
        super(credential);
        this.executorService = executorService;

    }

    /**
     * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、サーバーカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftySeparateAsyncClient(Credentials credential, ClientConfiguration configuration,
        ExecutorService executorService) {
        super(credential, configuration);
        this.executorService = executorService;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyCreateSeparateInstanceRuleResult>
        niftyCreateSeparateInstanceRuleAsync(final NiftyCreateSeparateInstanceRuleRequest request) {
        return executorService.submit(
            new Callable<NiftyCreateSeparateInstanceRuleResult>() {
                @Override
                public NiftyCreateSeparateInstanceRuleResult call() {
                    return niftyCreateSeparateInstanceRule(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDeleteSeparateInstanceRuleResult>
        niftyDeleteSeparateInstanceRuleAsync(final NiftyDeleteSeparateInstanceRuleRequest request) {
        return executorService.submit(
            new Callable<NiftyDeleteSeparateInstanceRuleResult>() {
                @Override
                public NiftyDeleteSeparateInstanceRuleResult call() {
                    return niftyDeleteSeparateInstanceRule(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDeregisterInstancesFromSeparateInstanceRuleResult>
        niftyDeregisterInstancesFromSeparateInstanceRuleAsync(
            final NiftyDeregisterInstancesFromSeparateInstanceRuleRequest request) {
        return executorService.submit(
            new Callable<NiftyDeregisterInstancesFromSeparateInstanceRuleResult>() {
                @Override
                public NiftyDeregisterInstancesFromSeparateInstanceRuleResult call() {
                    return niftyDeregisterInstancesFromSeparateInstanceRule(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDescribeSeparateInstanceRulesResult>
        niftyDescribeSeparateInstanceRulesAsync(
            final NiftyDescribeSeparateInstanceRulesRequest request) {
        return executorService.submit(
            new Callable<NiftyDescribeSeparateInstanceRulesResult>() {
                @Override
                public NiftyDescribeSeparateInstanceRulesResult call() {
                    return niftyDescribeSeparateInstanceRules(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyRegisterInstancesWithSeparateInstanceRuleResult>
        niftyRegisterInstancesWithSeparateInstanceRuleAsync(
            final NiftyRegisterInstancesWithSeparateInstanceRuleRequest request) {
        return executorService.submit(
            new Callable<NiftyRegisterInstancesWithSeparateInstanceRuleResult>() {
                @Override
                public NiftyRegisterInstancesWithSeparateInstanceRuleResult call() {
                    return niftyRegisterInstancesWithSeparateInstanceRule(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyUpdateSeparateInstanceRuleResult>
        niftyUpdateSeparateInstanceRuleAsync(final NiftyUpdateSeparateInstanceRuleRequest request) {
        return executorService.submit(
            new Callable<NiftyUpdateSeparateInstanceRuleResult>() {
                @Override
                public NiftyUpdateSeparateInstanceRuleResult call() {
                    return niftyUpdateSeparateInstanceRule(request);
                }
            });
    }
}
