/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.routetable.model.transform;

import com.nifty.cloud.sdk.routetable.model.*;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import java.util.ArrayList;
import java.util.List;

/**
 * RouteTableSetType変換クラス。<br />
 * このクラスはレスポンスXMLのrouteTableSet要素からRouteTableSetクラスのリストを生成します。
 */
public class RouteTableSetConverter implements Converter {

	private static final String ELEMENT_ROUTE_TABLE_ID = "routeTableId";
	private static final String ELEMENT_VPC_ID = "vpcId";
	private static final String ELEMENT_ROUTE_SET = "routeSet";
	private static final String ELEMENT_ASSOCIATION_SET = "associationSet";
	private static final String ELEMENT_PROPAGATING_VGW_SET = "propagatingVgwSet";
	private static final String ELEMENT_TAG_SET = "tagSet";

	/**
	 * 変換可能なクラスか判定します。
	 *
	 * @param clazz クラス
	 * @return 変換可能な場合true, 変換不可能な場合false
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public boolean canConvert(Class clazz) {
		Class[] interfaces = clazz.getInterfaces();
		for (Class i : interfaces) {
			if (i == List.class) {
				return true;
			}
		}
		return false;
	}

	/**
	 * {@literal
	 * routeTableSetをList<RouteTableSet>クラスに変換します。}
	 *
	 * @param reader XML読込リーダ
	 * @param context
	 * @return {@literal List<RouteTableSet>オブジェクト}
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader, UnmarshallingContext)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
		List<RouteTableSet> routeTableSets = new ArrayList<RouteTableSet>();
		String name = null;
		while(reader.hasMoreChildren()) {
			reader.moveDown();
			RouteTableSet routeTableSet = new RouteTableSet();

			while (reader.hasMoreChildren()) {
				reader.moveDown();

				Converter routeSetConverter = new RouteSetConverter();
				Converter associationSetConverter = new AssociationSetConverter();
				Converter propagatingVgwSetConverter = new PropagatingVgwSetConverter();
				Converter tagSetConverter = new TagSetConverter();

				name = reader.getNodeName();
				if (name.equals(ELEMENT_ROUTE_TABLE_ID)) {
					routeTableSet.setRouteTableId(reader.getValue());
				} else if (name.equals(ELEMENT_VPC_ID)) {
					routeTableSet.setVpcId(reader.getValue());
				} else if (name.equals(ELEMENT_ROUTE_SET)) {
					routeTableSet.setRoutes((List<Route>) routeSetConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_ASSOCIATION_SET)) {
					routeTableSet.setAssociations((List<RouteTableAssociation>) associationSetConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_PROPAGATING_VGW_SET)) {
					routeTableSet.setPropagatingVgws((List<PropagatingVgw>) propagatingVgwSetConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_TAG_SET)) {
					routeTableSet.setTagSet((List<Tag>) tagSetConverter.unmarshal(reader, context));
				}

				reader.moveUp();
			}
			routeTableSets.add(routeTableSet);
			reader.moveUp();
		}

		return routeTableSets;
	}

	/**
	 * 未実装メソッド
	 */
	@Override
	public void marshal(Object paramObject,
			HierarchicalStreamWriter paramHierarchicalStreamWriter,
			MarshallingContext paramMarshallingContext) {
	}
}
