/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.routetable;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.routetable.model.AssociateRouteTableRequest;
import com.nifty.cloud.sdk.routetable.model.AssociateRouteTableResult;
import com.nifty.cloud.sdk.routetable.model.CreateRouteRequest;
import com.nifty.cloud.sdk.routetable.model.CreateRouteResult;
import com.nifty.cloud.sdk.routetable.model.CreateRouteTableRequest;
import com.nifty.cloud.sdk.routetable.model.CreateRouteTableResult;
import com.nifty.cloud.sdk.routetable.model.DeleteRouteRequest;
import com.nifty.cloud.sdk.routetable.model.DeleteRouteResult;
import com.nifty.cloud.sdk.routetable.model.DeleteRouteTableRequest;
import com.nifty.cloud.sdk.routetable.model.DeleteRouteTableResult;
import com.nifty.cloud.sdk.routetable.model.DescribeRouteTablesRequest;
import com.nifty.cloud.sdk.routetable.model.DescribeRouteTablesResult;
import com.nifty.cloud.sdk.routetable.model.DisassociateRouteTableRequest;
import com.nifty.cloud.sdk.routetable.model.DisassociateRouteTableResult;
import com.nifty.cloud.sdk.routetable.model.ReplaceRouteRequest;
import com.nifty.cloud.sdk.routetable.model.ReplaceRouteResult;
import com.nifty.cloud.sdk.routetable.model.ReplaceRouteTableAssociationRequest;
import com.nifty.cloud.sdk.routetable.model.ReplaceRouteTableAssociationResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * ルートテーブルカテゴリクライアント。<br />
 * ルートテーブル カテゴリの非同期クライアントの具象クラスです。<br />
 * 戻り値の"利用不可"パラメータは返却値がnullまたは空となります。
 */
public class NiftyRouteTableAsyncClient extends NiftyRouteTableClient implements NiftyRouteTableAsync {


	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、ルートテーブルカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftyRouteTableAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報、クライアント設定を指定し、ルートテーブルカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftyRouteTableAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、ルートテーブルカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyRouteTableAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;

	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、ルートテーブルカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyRouteTableAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#createRouteTableAsync(com.nifty.cloud.sdk.routetable.model.CreateRouteTableRequest)
	 */
	@Override
	public Future<CreateRouteTableResult> createRouteTableAsync(final CreateRouteTableRequest request) {
		return executorService.submit(
				new Callable<CreateRouteTableResult>() {
					@Override
					public CreateRouteTableResult call() {
						return createRouteTable(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#deleteRouteTableAsync(com.nifty.cloud.sdk.routetable.model.DeleteRouteTableRequest)
	 */
	@Override
	public Future<DeleteRouteTableResult> deleteRouteTableAsync(final DeleteRouteTableRequest request) {
		return executorService.submit(
				new Callable<DeleteRouteTableResult>() {
					@Override
					public DeleteRouteTableResult call() {
						return deleteRouteTable(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#describeRouteTablesAsync(com.nifty.cloud.sdk.routetable.model.DescribeRouteTablesRequest)
	 */
	@Override
	public Future<DescribeRouteTablesResult> describeRouteTablesAsync(final DescribeRouteTablesRequest request) {
		return executorService.submit(
				new Callable<DescribeRouteTablesResult>() {
					@Override
					public DescribeRouteTablesResult call() {
						return describeRouteTables(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#associateRouteTableAsync(com.nifty.cloud.sdk.routetable.model.AssociateRouteTableRequest)
	 */
	@Override
	public Future<AssociateRouteTableResult> associateRouteTableAsync(final AssociateRouteTableRequest request) {
		return executorService.submit(
				new Callable<AssociateRouteTableResult>() {
					@Override
					public AssociateRouteTableResult call() {
						return associateRouteTable(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#disassociateRouteTableAsync(com.nifty.cloud.sdk.routetable.model.DisassociateRouteTableRequest)
	 */
	@Override
	public Future<DisassociateRouteTableResult> disassociateRouteTableAsync(final DisassociateRouteTableRequest request) {
		return executorService.submit(
				new Callable<DisassociateRouteTableResult>() {
					@Override
					public DisassociateRouteTableResult call() {
						return disassociateRouteTable(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#replaceRouteTableAssociationAsync(com.nifty.cloud.sdk.routetable.model.ReplaceRouteTableAssociationRequest)
	 */
	@Override
	public Future<ReplaceRouteTableAssociationResult> replaceRouteTableAssociationAsync(final ReplaceRouteTableAssociationRequest request) {
		return executorService.submit(
				new Callable<ReplaceRouteTableAssociationResult>() {
					@Override
					public ReplaceRouteTableAssociationResult call() {
						return replaceRouteTableAssociation(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#createRouteAsync(com.nifty.cloud.sdk.routetable.model.CreateRouteRequest)
	 */
	@Override
	public Future<CreateRouteResult> createRouteAsync(final CreateRouteRequest request) {
		return executorService.submit(
				new Callable<CreateRouteResult>() {
					@Override
					public CreateRouteResult call() {
						return createRoute(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#deleteRouteAsync(com.nifty.cloud.sdk.routetable.model.DeleteRouteRequest)
	 */
	@Override
	public Future<DeleteRouteResult> deleteRouteAsync(final DeleteRouteRequest request) {
		return executorService.submit(
				new Callable<DeleteRouteResult>() {
					@Override
					public DeleteRouteResult call() {
						return deleteRoute(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.routetable.NiftyRouteTableAsync#replaceRouteAsync(com.nifty.cloud.sdk.routetable.model.ReplaceRouteRequest)
	 */
	@Override
	public Future<ReplaceRouteResult> replaceRouteAsync(final ReplaceRouteRequest request) {
		return executorService.submit(
				new Callable<ReplaceRouteResult>() {
					@Override
					public ReplaceRouteResult call() {
						return replaceRoute(request);
					}
				}
		);
	}
}