/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.router.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Request;
import com.nifty.cloud.sdk.annotation.Action;
import com.nifty.cloud.sdk.annotation.Query;

/**
 * NiftyUpdateRouterNetworkInterfacesリクエストクラス。<br />
 * このクラスはNiftyUpdateRouterNetworkInterfacesへのリクエストを構築します。
 *
 */
@Action("NiftyUpdateRouterNetworkInterfaces")
public class NiftyUpdateRouterNetworkInterfacesRequest implements Request {

	/** ルーターユニークID */
	private String routerId;

	/** ルーター名 */
	private String routerName;

	/** ネットワークインターフェイス情報リスト */
	private List<NetworkInterface> networkInterfaces;

	/** 再起動オプション */
	private String niftyReboot;

	/** version up実施したのち、バックアップ状態を解除する事項への同意 */
	private Boolean agreement;

	/**
	 * デフォルトコンストラクタ。
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest() {
	}

	/**
	 * ネットワークインターフェイス情報リストを指定し、NiftyUpdateRouterNetworkInterfacesRequestを構築します。
	 *
	 * @param networkInterfaces ネットワークインターフェイス情報リスト
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest(List<NetworkInterface> networkInterfaces) {
		this.networkInterfaces = networkInterfaces;
	}

	/**
	 * ルーターユニークIDを取得します。
	 *
	 * @return ルーターユニークID
	 */
	@Query(name="RouterId")
	public String getRouterId() {
		return routerId;
	}

	/**
	 * ルーターユニークIDを設定します。
	 *
	 * @param routerId ルーターユニークID
	 */
	public void setRouterId(String routerId) {
		this.routerId = routerId;
	}

	/**
	 * ルーターユニークIDを設定し、自オブジェクトを返します。
	 *
	 * @param routerId ルーターユニークID
	 * @return 自オブジェクト
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest withRouterId(String routerId) {
		setRouterId(routerId);
		return this;
	}

	/**
	 * ルーター名を取得します。
	 *
	 * @return ルーター名
	 */
	@Query(name="RouterName")
	public String getRouterName() {
		return routerName;
	}

	/**
	 * ルーター名を設定します。
	 *
	 * @param routerName ルーター名
	 */
	public void setRouterName(String routerName) {
		this.routerName = routerName;
	}

	/**
	 * ルーター名を設定し、自オブジェクトを返します。
	 *
	 * @param routerName ルーター名
	 * @return 自オブジェクト
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest withRouterName(String routerName) {
		setRouterName(routerName);
		return this;
	}

	/**
	 * ネットワークインターフェイス情報リストを取得します。
	 *
	 * @return ネットワークインターフェイス情報リスト
	 */
	@Query(name="NetworkInterface", require=true)
	public List<NetworkInterface> getNetworkInterfaces() {
		return networkInterfaces;
	}

	/**
	 * ネットワークインターフェイス情報リストを設定します。
	 *
	 * @param networkInterfaces ネットワークインターフェイス情報リスト
	 */
	public void setNetworkInterfaces(List<NetworkInterface> networkInterfaces) {
		this.networkInterfaces = networkInterfaces;
	}

	/**
	 * ネットワークインターフェイス情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param networkInterfaces ネットワークインターフェイス情報の配列
	 * @return 自オブジェクト
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest withNetworkInterfaces(NetworkInterface ... networkInterfaces) {
		if (this.networkInterfaces == null) this.networkInterfaces = new ArrayList<NetworkInterface>();
		for (NetworkInterface networkInterface : networkInterfaces) {
			getNetworkInterfaces().add(networkInterface);
		}
		return this;
	}

	/**
	 * ネットワークインターフェイス情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param networkInterfaces ネットワークインターフェイス情報リスト
	 * @return 自オブジェクト
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest withNetworkInterfaces(Collection<NetworkInterface> networkInterfaces) {
		if (this.networkInterfaces == null) this.networkInterfaces = new ArrayList<NetworkInterface>();
		if (networkInterfaces != null) {
			getNetworkInterfaces().addAll(networkInterfaces);
		}
		return this;
	}

	/**
	 * 再起動オプションを取得します。
	 *
	 * @return 再起動オプション
	 */
	@Query(name="NiftyReboot")
	public String getNiftyReboot() {
		return niftyReboot;
	}

	/**
	 * 再起動オプションを設定します。
	 *
	 * @param niftyReboot 再起動オプション
	 */
	public void setNiftyReboot(String niftyReboot) {
		this.niftyReboot = niftyReboot;
	}

	/**
	 * 再起動オプションを設定し、自オブジェクトを返します。
	 *
	 * @param niftyReboot 再起動オプション
	 * @return 自オブジェクト
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest withNiftyReboot(String niftyReboot) {
		setNiftyReboot(niftyReboot);
		return this;
	}

	/**
	 * version up実施したのち、バックアップ状態を解除する事項への同意を取得します。
	 *
	 * @return version up実施したのち、バックアップ状態を解除する事項への同意
	 */
	@Query(name="Agreement")
	public Boolean getAgreement() {
		return agreement;
	}

	/**
	 * version up実施したのち、バックアップ状態を解除する事項への同意を設定します。
	 *
	 * @param agreement version up実施したのち、バックアップ状態を解除する事項への同意
	 */
	public void setAgreement(Boolean agreement) {
		this.agreement = agreement;
	}

	/**
	 * version up実施したのち、バックアップ状態を解除する事項への同意を設定し、自オブジェクトを返します。
	 *
	 * @param agreement version up実施したのち、バックアップ状態を解除する事項への同意
	 * @return 自オブジェクト
	 */
	public NiftyUpdateRouterNetworkInterfacesRequest withAgreement(Boolean agreement) {
		setAgreement(agreement);
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[routerId=");
		builder.append(routerId);
		builder.append(", routerName=");
		builder.append(routerName);
		builder.append(", networkInterfaces=");
		builder.append(networkInterfaces);
		builder.append(", niftyReboot=");
		builder.append(niftyReboot);
		builder.append(", agreement=");
		builder.append(agreement);
		builder.append("]");
		return builder.toString();
	}
}
