/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.router;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.router.model.NiftyCreateRouterRequest;
import com.nifty.cloud.sdk.router.model.NiftyCreateRouterResult;
import com.nifty.cloud.sdk.router.model.NiftyDeleteRouterRequest;
import com.nifty.cloud.sdk.router.model.NiftyDeleteRouterResult;
import com.nifty.cloud.sdk.router.model.NiftyDescribeRoutersRequest;
import com.nifty.cloud.sdk.router.model.NiftyDescribeRoutersResult;
import com.nifty.cloud.sdk.router.model.NiftyModifyRouterAttributeRequest;
import com.nifty.cloud.sdk.router.model.NiftyModifyRouterAttributeResult;
import com.nifty.cloud.sdk.router.model.NiftyRebootRoutersRequest;
import com.nifty.cloud.sdk.router.model.NiftyRebootRoutersResult;
import com.nifty.cloud.sdk.router.model.NiftyReleaseRouterBackupStateRequest;
import com.nifty.cloud.sdk.router.model.NiftyReleaseRouterBackupStateResult;
import com.nifty.cloud.sdk.router.model.NiftyReplaceRouterLatestVersionRequest;
import com.nifty.cloud.sdk.router.model.NiftyReplaceRouterLatestVersionResult;
import com.nifty.cloud.sdk.router.model.NiftyRestoreRouterPreviousVersionRequest;
import com.nifty.cloud.sdk.router.model.NiftyRestoreRouterPreviousVersionResult;
import com.nifty.cloud.sdk.router.model.NiftyUpdateRouterNetworkInterfacesRequest;
import com.nifty.cloud.sdk.router.model.NiftyUpdateRouterNetworkInterfacesResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * ルーターカテゴリクライアント。<br />
 * ルーター カテゴリの非同期クライアントの具象クラスです。<br />
 * 戻り値の"利用不可"パラメータは返却値がnullまたは空となります。
 *
 */
public class NiftyRouterAsyncClient extends NiftyRouterClient implements NiftyRouterAsync {


	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、ルーターカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftyRouterAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報、クライアント設定を指定し、ルーターカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftyRouterAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、ルーターカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyRouterAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;

	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、ルーターカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyRouterAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyCreateRouterAsync(com.nifty.cloud.sdk.router.model.NiftyCreateRouterRequest)
	 */
	@Override
	public Future<NiftyCreateRouterResult> niftyCreateRouterAsync(final NiftyCreateRouterRequest request) {
		return executorService.submit(
				new Callable<NiftyCreateRouterResult>() {
					@Override
					public NiftyCreateRouterResult call() {
						return niftyCreateRouter(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyDeleteRouterAsync(com.nifty.cloud.sdk.router.model.NiftyDeleteRouterRequest)
	 */
	@Override
	public Future<NiftyDeleteRouterResult> niftyDeleteRouterAsync(final NiftyDeleteRouterRequest request) {
		return executorService.submit(
				new Callable<NiftyDeleteRouterResult>() {
					@Override
					public NiftyDeleteRouterResult call() {
						return niftyDeleteRouter(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyDescribeRoutersAsync(com.nifty.cloud.sdk.router.model.NiftyDescribeRoutersRequest)
	 */
	@Override
	public Future<NiftyDescribeRoutersResult> niftyDescribeRoutersAsync(final NiftyDescribeRoutersRequest request) {
		return executorService.submit(
				new Callable<NiftyDescribeRoutersResult>() {
					@Override
					public NiftyDescribeRoutersResult call() {
						return niftyDescribeRouters(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyRebootRoutersAsync(com.nifty.cloud.sdk.router.model.NiftyRebootRoutersRequest)
	 */
	@Override
	public Future<NiftyRebootRoutersResult> niftyRebootRoutersAsync(final NiftyRebootRoutersRequest request) {
		return executorService.submit(
				new Callable<NiftyRebootRoutersResult>() {
					@Override
					public NiftyRebootRoutersResult call() {
						return niftyRebootRouters(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyModifyRouterAttributeAsync(com.nifty.cloud.sdk.router.model.NiftyModifyRouterAttributeRequest)
	 */
	@Override
	public Future<NiftyModifyRouterAttributeResult> niftyModifyRouterAttributeAsync(final NiftyModifyRouterAttributeRequest request) {
		return executorService.submit(
				new Callable<NiftyModifyRouterAttributeResult>() {
					@Override
					public NiftyModifyRouterAttributeResult call() {
						return niftyModifyRouterAttribute(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyUpdateRouterNetworkInterfacesAsync(com.nifty.cloud.sdk.router.model.NiftyUpdateRouterNetworkInterfacesRequest)
	 */
	@Override
	public Future<NiftyUpdateRouterNetworkInterfacesResult> niftyUpdateRouterNetworkInterfacesAsync(final NiftyUpdateRouterNetworkInterfacesRequest request) {
		return executorService.submit(
				new Callable<NiftyUpdateRouterNetworkInterfacesResult>() {
					@Override
					public NiftyUpdateRouterNetworkInterfacesResult call() {
						return niftyUpdateRouterNetworkInterfaces(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyReplaceRouterLatestVersionAsync(com.nifty.cloud.sdk.router.model.NiftyReplaceRouterLatestVersionRequest)
	 */
	@Override
	public Future<NiftyReplaceRouterLatestVersionResult> niftyReplaceRouterLatestVersionAsync(final NiftyReplaceRouterLatestVersionRequest request) {
		return executorService.submit(
				new Callable<NiftyReplaceRouterLatestVersionResult>() {
					@Override
					public NiftyReplaceRouterLatestVersionResult call() {
						return niftyReplaceRouterLatestVersion(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyRestoreRouterPreviousVersionAsync(com.nifty.cloud.sdk.router.model.NiftyRestoreRouterPreviousVersionRequest)
	 */
	@Override
	public Future<NiftyRestoreRouterPreviousVersionResult> niftyRestoreRouterPreviousVersionAsync(final NiftyRestoreRouterPreviousVersionRequest request) {
		return executorService.submit(
				new Callable<NiftyRestoreRouterPreviousVersionResult>() {
					@Override
					public NiftyRestoreRouterPreviousVersionResult call() {
						return niftyRestoreRouterPreviousVersion(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.router.NiftyRouterAsync#niftyReleaseRouterBackupStateAsync(com.nifty.cloud.sdk.router.model.NiftyReleaseRouterBackupStateRequest)
	 */
	@Override
	public Future<NiftyReleaseRouterBackupStateResult> niftyReleaseRouterBackupStateAsync(final NiftyReleaseRouterBackupStateRequest request) {
		return executorService.submit(
				new Callable<NiftyReleaseRouterBackupStateResult>() {
					@Override
					public NiftyReleaseRouterBackupStateResult call() {
						return niftyReleaseRouterBackupState(request);
					}
				}
		);
	}
}