/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.proxy.model.transform;

import com.nifty.cloud.sdk.proxy.model.NetworkInterface;
import com.nifty.cloud.sdk.proxy.model.WebProxy;
import com.nifty.cloud.sdk.proxy.model.WebProxyOption;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * WebProxyType変換クラス。<br />
 * このクラスはレスポンスXMLのwebProxy要素からWebProxyクラスを生成します。
 */
public class WebProxyConverter implements Converter {

	private static final String ELEMENT_ROUTER_ID = "routerId";
	private static final String ELEMENT_ROUTER_NAME = "routerName";
	private static final String ELEMENT_LISTEN_INTERFACE = "listenInterface";
	private static final String ELEMENT_LISTEN_PORT = "listenPort";
	private static final String ELEMENT_BYPASS_INTERFACE = "bypassInterface";
	private static final String ELEMENT_OPTION = "option";
	private static final String ELEMENT_DESCRIPTION = "description";

	/**
	 * 変換可能なクラスか判定します。
	 *
	 * @param clazz クラス
	 * @return 変換可能な場合true, 変換不可能な場合false
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public boolean canConvert(Class clazz) {
		return clazz == WebProxy.class;
	}

	/**
	 * {@literal
	 * webProxyをWebProxyクラスに変換します。}
	 *
	 * @param reader XML読込リーダ
	 * @param context
	 * @return {@literal WebProxyオブジェクト}
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader, UnmarshallingContext)
	 */
	@Override
	public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
		WebProxy webProxy = new WebProxy();
		Converter networkInterfaceConverter = new NetworkInterfaceConverter();
		Converter optionConverter = new OptionConverter();

		String name = null;
		while (reader.hasMoreChildren()) {
			reader.moveDown();

			name = reader.getNodeName();
			if (name.equals(ELEMENT_ROUTER_ID)) {
				webProxy.setRouterId(reader.getValue());
			} else if (name.equals(ELEMENT_ROUTER_NAME)) {
				webProxy.setRouterName(reader.getValue());
			} else if (name.equals(ELEMENT_LISTEN_INTERFACE)) {
				webProxy.setListenInterface((NetworkInterface) networkInterfaceConverter.unmarshal(reader, context));
			} else if (name.equals(ELEMENT_LISTEN_PORT)) {
				webProxy.setListenPort(reader.getValue());
			} else if (name.equals(ELEMENT_BYPASS_INTERFACE)) {
				webProxy.setBypassInterface((NetworkInterface) networkInterfaceConverter.unmarshal(reader, context));
			} else if (name.equals(ELEMENT_OPTION)) {
				webProxy.setOption((WebProxyOption) optionConverter.unmarshal(reader, context));
			} else if (name.equals(ELEMENT_DESCRIPTION)) {
				webProxy.setDescription(reader.getValue());
			}

			reader.moveUp();
		}

		return webProxy;
	}

	/**
	 * 未実装メソッド
	 */
	@Override
	public void marshal(Object paramObject,
			HierarchicalStreamWriter paramHierarchicalStreamWriter,
			MarshallingContext paramMarshallingContext) {
	}
}
