/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.proxy.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Result;
import com.nifty.cloud.sdk.proxy.model.transform.NiftyDescribeWebProxiesResultConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * NiftyDescribeWebProxies処理結果クラス。<br />
 * このクラスはNiftyDescribeWebProxiesのレスポンスを格納します。
 */
@XStreamAlias("NiftyDescribeWebProxiesResponse")
@XStreamConverter(NiftyDescribeWebProxiesResultConverter.class)
public class NiftyDescribeWebProxiesResult extends Result {

	/** リクエスト識別子 */
	private String requestId;

	/** Webプロキシ情報リスト */
	private List<WebProxy> webProxies;

	/**
	 * リクエスト識別子を取得します。
	 *
	 * @return リクエスト識別子
	 */
	public String getRequestId() {
		return requestId;
	}

	/**
	 * リクエスト識別子を設定します。
	 *
	 * @param requestId リクエスト識別子
	 */
	public void setRequestId(String requestId) {
		this.requestId = requestId;
	}

	/**
	 * リクエスト識別子を設定し、自オブジェクトを返します。
	 *
	 * @param requestId リクエスト識別子
	 * @return 自オブジェクト
	 */
	public NiftyDescribeWebProxiesResult withRequestId(String requestId) {
		setRequestId(requestId);
		return this;
	}

	/**
	 * Webプロキシ情報リストを取得します。
	 *
	 * @return Webプロキシ情報リスト
	 */
	public List<WebProxy> getWebProxies() {
		return webProxies;
	}

	/**
	 * Webプロキシ情報リストを設定します。
	 *
	 * @param webProxies Webプロキシ情報リスト
	 */
	public void setWebProxies(List<WebProxy> webProxies) {
		this.webProxies = webProxies;
	}

	/**
	 * Webプロキシ情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param webProxies Webプロキシ情報の配列
	 * @return 自オブジェクト
	 */
	public NiftyDescribeWebProxiesResult withWebProxies(WebProxy ... webProxies) {
		if (this.webProxies == null) this.webProxies = new ArrayList<WebProxy>();
		for (WebProxy webProxy : webProxies) {
			getWebProxies().add(webProxy);
		}
		return this;
	}

	/**
	 * Webプロキシ情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param webProxies Webプロキシ情報リスト
	 * @return 自オブジェクト
	 */
	public NiftyDescribeWebProxiesResult withWebProxies(Collection<WebProxy> webProxies) {
		if (this.webProxies == null) this.webProxies = new ArrayList<WebProxy>();
		if (webProxies != null) {
			getWebProxies().addAll(webProxies);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[requestId=");
		builder.append(requestId);
		builder.append(", webProxies=");
		builder.append(webProxies);
		builder.append("]");
		return builder.toString();
	}
}
