/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.proxy;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.proxy.model.NiftyCreateWebProxyRequest;
import com.nifty.cloud.sdk.proxy.model.NiftyCreateWebProxyResult;
import com.nifty.cloud.sdk.proxy.model.NiftyDeleteWebProxyRequest;
import com.nifty.cloud.sdk.proxy.model.NiftyDeleteWebProxyResult;
import com.nifty.cloud.sdk.proxy.model.NiftyDescribeWebProxiesRequest;
import com.nifty.cloud.sdk.proxy.model.NiftyDescribeWebProxiesResult;
import com.nifty.cloud.sdk.proxy.model.NiftyModifyWebProxyAttributeRequest;
import com.nifty.cloud.sdk.proxy.model.NiftyModifyWebProxyAttributeResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * プロキシカテゴリクライアント。<br />
 * プロキシ カテゴリの非同期クライアントの具象クラスです。
 *
 */
public class NiftyProxyAsyncClient extends NiftyProxyClient implements NiftyProxyAsync {


	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、プロキシカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftyProxyAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報、クライアント設定を指定し、プロキシカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftyProxyAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、プロキシカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyProxyAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;

	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、プロキシカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyProxyAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.proxy.NiftyProxyAsync#niftyCreateWebProxyAsync(com.nifty.cloud.sdk.proxy.model.NiftyCreateWebProxyRequest)
	 */
	@Override
	public Future<NiftyCreateWebProxyResult> niftyCreateWebProxyAsync(final NiftyCreateWebProxyRequest request) {
		return executorService.submit(
				new Callable<NiftyCreateWebProxyResult>() {
					@Override
					public NiftyCreateWebProxyResult call() {
						return niftyCreateWebProxy(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.proxy.NiftyProxyAsync#niftyDeleteWebProxyAsync(com.nifty.cloud.sdk.proxy.model.NiftyDeleteWebProxyRequest)
	 */
	@Override
	public Future<NiftyDeleteWebProxyResult> niftyDeleteWebProxyAsync(final NiftyDeleteWebProxyRequest request) {
		return executorService.submit(
				new Callable<NiftyDeleteWebProxyResult>() {
					@Override
					public NiftyDeleteWebProxyResult call() {
						return niftyDeleteWebProxy(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.proxy.NiftyProxyAsync#niftyDescribeWebProxiesAsync(com.nifty.cloud.sdk.proxy.model.NiftyDescribeWebProxiesRequest)
	 */
	@Override
	public Future<NiftyDescribeWebProxiesResult> niftyDescribeWebProxiesAsync(final NiftyDescribeWebProxiesRequest request) {
		return executorService.submit(
				new Callable<NiftyDescribeWebProxiesResult>() {
					@Override
					public NiftyDescribeWebProxiesResult call() {
						return niftyDescribeWebProxies(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.proxy.NiftyProxyAsync#niftyModifyWebProxyAttributeAsync(com.nifty.cloud.sdk.proxy.model.NiftyModifyWebProxyAttributeRequest)
	 */
	@Override
	public Future<NiftyModifyWebProxyAttributeResult> niftyModifyWebProxyAttributeAsync(final NiftyModifyWebProxyAttributeRequest request) {
		return executorService.submit(
				new Callable<NiftyModifyWebProxyAttributeResult>() {
					@Override
					public NiftyModifyWebProxyAttributeResult call() {
						return niftyModifyWebProxyAttribute(request);
					}
				}
		);
	}
}