/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.privatelan;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.privatelan.model.NiftyCreatePrivateLanRequest;
import com.nifty.cloud.sdk.privatelan.model.NiftyCreatePrivateLanResult;
import com.nifty.cloud.sdk.privatelan.model.NiftyDeletePrivateLanRequest;
import com.nifty.cloud.sdk.privatelan.model.NiftyDeletePrivateLanResult;
import com.nifty.cloud.sdk.privatelan.model.NiftyDescribePrivateLansRequest;
import com.nifty.cloud.sdk.privatelan.model.NiftyDescribePrivateLansResult;
import com.nifty.cloud.sdk.privatelan.model.NiftyModifyPrivateLanAttributeRequest;
import com.nifty.cloud.sdk.privatelan.model.NiftyModifyPrivateLanAttributeResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * プライベートLANカテゴリクライアント。<br />
 * プライベートLANカテゴリの非同期クライアントの具象クラスです。<br />
 * 戻り値の"利用不可"パラメータは返却値がnullまたは空となります。
 */
public class NiftyPrivateLanAsyncClient extends NiftyPrivateLanClient implements NiftyPrivateLanAsync {

	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、プライベートLANカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftyPrivateLanAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,クライアント情報を指定し、プライベートLANカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftyPrivateLanAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、プライベートLANカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyPrivateLanAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;
	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、プライベートLANカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyPrivateLanAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.privatelan.NiftyPrivateLanAsync#niftyCreatePrivateLanAsync(com.nifty.cloud.sdk.privatelan.model.NiftyCreatePrivateLanRequest)
	 */
	@Override
	public Future<NiftyCreatePrivateLanResult> niftyCreatePrivateLanAsync(final NiftyCreatePrivateLanRequest request) {
		return executorService.submit(
				new Callable<NiftyCreatePrivateLanResult>() {
					@Override
					public NiftyCreatePrivateLanResult call() {
						return niftyCreatePrivateLan(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.privatelan.NiftyPrivateLanAsync#niftyDeletePrivateLanAsync(com.nifty.cloud.sdk.privatelan.model.NiftyDeletePrivateLanRequest)
	 */
	@Override
	public Future<NiftyDeletePrivateLanResult> niftyDeletePrivateLanAsync(final NiftyDeletePrivateLanRequest request) {
		return executorService.submit(
				new Callable<NiftyDeletePrivateLanResult>() {
					@Override
					public NiftyDeletePrivateLanResult call() {
						return niftyDeletePrivateLan(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.privatelan.NiftyPrivateLansAsync#niftyDescribePrivateLansAsync(com.nifty.cloud.sdk.privatelan.model.NiftyDescribePrivateLansRequest)
	 */
	@Override
	public Future<NiftyDescribePrivateLansResult> niftyDescribePrivateLansAsync(final NiftyDescribePrivateLansRequest request) {
		return executorService.submit(
				new Callable<NiftyDescribePrivateLansResult>() {
					@Override
					public NiftyDescribePrivateLansResult call() {
						return niftyDescribePrivateLans(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.privatelan.NiftyPrivateLanAsync#niftyModifyPrivateLanAttributeAsync(com.nifty.cloud.sdk.privatelan.model.NiftyModifyPrivateLanAttributeRequest)
	 */
	@Override
	public Future<NiftyModifyPrivateLanAttributeResult> niftyModifyPrivateLanAttributeAsync(final NiftyModifyPrivateLanAttributeRequest request) {
		return executorService.submit(
				new Callable<NiftyModifyPrivateLanAttributeResult>() {
					@Override
					public NiftyModifyPrivateLanAttributeResult call() {
						return niftyModifyPrivateLanAttribute(request);
					}
				}
		);
	}
}
