/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.nic.model.transform;

import com.nifty.cloud.sdk.nic.model.Attachment;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * Attachment変換クラス。<br />
 * このクラスはレスポンスXMLのattachment要素からAttachmentクラスを生成します。
 *
 */
public class AttachmentConverter implements Converter {

    private static final String ELEMENT_ATTACHMENT_ID = "attachmentId";
    private static final String ELEMENT_ATTACH_TIME = "attachTime";
    private static final String ELEMENT_DELETE_ON_TERMINATION = "deleteOnTermination";
    private static final String ELEMENT_DEVICE_INDEX = "deviceIndex";
    private static final String ELEMENT_INSTANCE_ID = "instanceId";
    private static final String ELEMENT_INSTANCE_OWNER_ID = "instanceOwnerId";
    private static final String ELEMENT_STATUS = "status";

    /**
     * 変換可能なクラスか判定します。
     *
     * @param clazz
     *            クラス
     * @return 変換可能な場合true, 変換不可能な場合false
     *
     * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
     */
    @SuppressWarnings("rawtypes")
    @Override
    public boolean canConvert(Class clazz) {
        return clazz == Attachment.class;
    }

    /**
     * {@literal
     * AttachmentをAttachmentクラスに変換します。}
     *
     * @param reader
     *            XML読込リーダ
     * @param context
     * @return {@literal Attachmentオブジェクト}
     *
     * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader,
     *      UnmarshallingContext)
     */
    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        Attachment attachment = new Attachment();

        String name = null;
        while (reader.hasMoreChildren()) {
            reader.moveDown();

            name = reader.getNodeName();
            if (name.equals(ELEMENT_ATTACHMENT_ID)) {
                attachment.setAttachmentId(reader.getValue());
            } else if (name.equals(ELEMENT_ATTACH_TIME)) {
                attachment.setAttachTime(reader.getValue());
            } else if (name.equals(ELEMENT_DELETE_ON_TERMINATION)) {
                attachment.setDeleteOnTermination(Boolean.valueOf(reader.getValue()));
            } else if (name.equals(ELEMENT_DEVICE_INDEX)) {
                attachment.setDeviceIndex(reader.getValue());
            } else if (name.equals(ELEMENT_INSTANCE_ID)) {
                attachment.setInstanceId(reader.getValue());
            } else if (name.equals(ELEMENT_INSTANCE_OWNER_ID)) {
                attachment.setInstanceOwnerId(reader.getValue());
            } else if (name.equals(ELEMENT_STATUS)) {
                attachment.setStatus(reader.getValue());
            }

            reader.moveUp();
        }

        return attachment;
    }

    /**
     * 未実装メソッド
     */
    @Override
    public void marshal(Object paramObject,
        HierarchicalStreamWriter paramHierarchicalStreamWriter,
        MarshallingContext paramMarshallingContext) {
        // 処理無し
    }
}
