/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.nic;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.nic.model.AttachNetworkInterfaceRequest;
import com.nifty.cloud.sdk.nic.model.AttachNetworkInterfaceResult;
import com.nifty.cloud.sdk.nic.model.CreateNetworkInterfaceRequest;
import com.nifty.cloud.sdk.nic.model.CreateNetworkInterfaceResult;
import com.nifty.cloud.sdk.nic.model.DeleteNetworkInterfaceRequest;
import com.nifty.cloud.sdk.nic.model.DeleteNetworkInterfaceResult;
import com.nifty.cloud.sdk.nic.model.DescribeNetworkInterfacesRequest;
import com.nifty.cloud.sdk.nic.model.DescribeNetworkInterfacesResult;
import com.nifty.cloud.sdk.nic.model.DetachNetworkInterfaceRequest;
import com.nifty.cloud.sdk.nic.model.DetachNetworkInterfaceResult;
import com.nifty.cloud.sdk.nic.model.ModifyNetworkInterfaceAttributeRequest;
import com.nifty.cloud.sdk.nic.model.ModifyNetworkInterfaceAttributeResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。 追加NICカテゴリクライアント。<br />
 * 追加NICカテゴリの非同期クライアントの具象クラスです。
 */
public class NiftyNicAsyncClient extends NiftyNicClient implements NiftyNicAsync {

    /** 非同期タスク実行クラス. */
    protected ExecutorService executorService = null;

    /**
     * 認証情報を指定し、追加NICカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     */
    public NiftyNicAsyncClient(Credentials credential) {
        this(credential, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報,クライアント情報を指定し、追加NICカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     */
    public NiftyNicAsyncClient(Credentials credential, ClientConfiguration configuration) {
        this(credential, configuration, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報,非同期タスク実行クラスを指定し、追加NICカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftyNicAsyncClient(Credentials credential, ClientConfiguration configuration,
        ExecutorService executorService) {
        super(credential, configuration);
        this.executorService = executorService;
    }

    /**
     * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、追加NICカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftyNicAsyncClient(Credentials credential, ExecutorService executorService) {
        super(credential);
        this.executorService = executorService;

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<CreateNetworkInterfaceResult>
        createNetworkInterfaceAsync(final CreateNetworkInterfaceRequest request) {
        return executorService.submit(
            new Callable<CreateNetworkInterfaceResult>() {
                @Override
                public CreateNetworkInterfaceResult call() {
                    return createNetworkInterface(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<DescribeNetworkInterfacesResult>
        describeNetworkInterfacesAsync(final DescribeNetworkInterfacesRequest request) {
        return executorService.submit(
            new Callable<DescribeNetworkInterfacesResult>() {
                @Override
                public DescribeNetworkInterfacesResult call() {
                    return describeNetworkInterfaces(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<AttachNetworkInterfaceResult>
        attachNetworkInterfaceAsync(final AttachNetworkInterfaceRequest request) {
        return executorService.submit(
            new Callable<AttachNetworkInterfaceResult>() {
                @Override
                public AttachNetworkInterfaceResult call() {
                    return attachNetworkInterface(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<DetachNetworkInterfaceResult>
        detachNetworkInterfaceAsync(final DetachNetworkInterfaceRequest request) {
        return executorService.submit(
            new Callable<DetachNetworkInterfaceResult>() {
                @Override
                public DetachNetworkInterfaceResult call() {
                    return detachNetworkInterface(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<DeleteNetworkInterfaceResult>
        deleteNetworkInterfaceAsync(final DeleteNetworkInterfaceRequest request) {
        return executorService.submit(
            new Callable<DeleteNetworkInterfaceResult>() {
                @Override
                public DeleteNetworkInterfaceResult call() {
                    return deleteNetworkInterface(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<ModifyNetworkInterfaceAttributeResult>
        modifyNetworkInterfaceAttributeAsync(final ModifyNetworkInterfaceAttributeRequest request) {
        return executorService.submit(
            new Callable<ModifyNetworkInterfaceAttributeResult>() {
                @Override
                public ModifyNetworkInterfaceAttributeResult call() {
                    return modifyNetworkInterfaceAttribute(request);
                }
            });
    }
}
