/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.nattable.model.transform;

import java.util.ArrayList;
import java.util.List;

import com.nifty.cloud.sdk.nattable.model.Destination;
import com.nifty.cloud.sdk.nattable.model.NatRule;
import com.nifty.cloud.sdk.nattable.model.NetworkInterface;
import com.nifty.cloud.sdk.nattable.model.Source;
import com.nifty.cloud.sdk.nattable.model.Translation;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * NatRuleSetType変換クラス。<br />
 * このクラスはレスポンスXMLのnatRuleSet要素からNatRuleクラスのリストを生成します。
 */
public class NatRuleSetConverter implements Converter {

	private static final String ELEMENT_NAT_TYPE = "natType";
	private static final String ELEMENT_RULE_NUMBER = "ruleNumber";
	private static final String ELEMENT_DESCRIPTION = "description";
	private static final String ELEMENT_PROTOCOL = "protocol";
	private static final String ELEMENT_OUTBOUND_INTERFACE = "outboundInterface";
	private static final String ELEMENT_INBOUND_INTERFACE = "inboundInterface";
	private static final String ELEMENT_DESTINATION = "destination";
	private static final String ELEMENT_SOURCE = "source";
	private static final String ELEMENT_TRANSLATION = "translation";

	/**
	 * 変換可能なクラスか判定します。
	 *
	 * @param clazz クラス
	 * @return 変換可能な場合true, 変換不可能な場合false
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public boolean canConvert(Class clazz) {
		Class[] interfaces = clazz.getInterfaces();
		for (Class i : interfaces) {
			if (i == List.class) {
				return true;
			}
		}
		return false;
	}

	/**
	 * {@literal
	 * natRuleSetをList<NatRule>クラスに変換します。}
	 *
	 * @param reader XML読込リーダ
	 * @param context
	 * @return {@literal List<NatRule>オブジェクト}
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader, UnmarshallingContext)
	 */
	@Override
	public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
		List<NatRule> natRules = new ArrayList<NatRule>();
		Converter networkInterfaceConverter = new NetworkInterfaceConverter();
		Converter destinationConverter = new DestinationConverter();
		Converter sourceConverter = new SourceConverter();
		Converter translationConverter = new TranslationConverter();

		String name = null;
		while (reader.hasMoreChildren()) {
			reader.moveDown();
			NatRule natRule = new NatRule();

			while (reader.hasMoreChildren()) {
				reader.moveDown();

				name = reader.getNodeName();
				if (name.equals(ELEMENT_NAT_TYPE)) {
					natRule.setNatType(reader.getValue());
				} else if (name.equals(ELEMENT_RULE_NUMBER)) {
					natRule.setRuleNumber(reader.getValue());
				} else if (name.equals(ELEMENT_DESCRIPTION)) {
					natRule.setDescription(reader.getValue());
				} else if (name.equals(ELEMENT_PROTOCOL)) {
					natRule.setProtocol(reader.getValue());
				} else if (name.equals(ELEMENT_OUTBOUND_INTERFACE)) {
					natRule.setOutboundInterface((NetworkInterface) networkInterfaceConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_INBOUND_INTERFACE)) {
					natRule.setInboundInterface((NetworkInterface) networkInterfaceConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_DESTINATION)) {
					natRule.setDestination((Destination) destinationConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_SOURCE)) {
					natRule.setSource((Source) sourceConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_TRANSLATION)) {
					natRule.setTranslation((Translation) translationConverter.unmarshal(reader, context));
				}
				reader.moveUp();
			}
			reader.moveUp();
			natRules.add(natRule);
		}

		return natRules;
	}

	/**
	 * 未実装メソッド
	 */
	@Override
	public void marshal(Object paramObject,
			HierarchicalStreamWriter paramHierarchicalStreamWriter,
			MarshallingContext paramMarshallingContext) {
	}
}
