/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.image.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Request;
import com.nifty.cloud.sdk.annotation.Action;
import com.nifty.cloud.sdk.annotation.Query;

/**
 * NiftyAssociateImageリクエストクラス。<br />
 * このクラスはNiftyAssociateImageへのリクエストを構築します。
 */
@Action("NiftyAssociateImage")
public class NiftyAssociateImageRequest implements Request {

    /** OSイメージID */
    private String imageId;

    /** 公開レベル */
    private boolean isPublic;

    /** 再配布設定 */
    private boolean isRedistribute;

    /** 配布先@nifty 法人ID */
    private List<String> distributionIds;

    /**
     * デフォルトコンストラクタ。
     */
    public NiftyAssociateImageRequest() {
    }

    /**
     * OSイメージIDを指定し、NiftyAssociateImageRequestを構築します。
     *
     * @param imageId
     *            OSイメージID
     */
    public NiftyAssociateImageRequest(String imageId) {
        this.imageId = imageId;
    }

    /**
     * OSイメージIDを取得します。
     *
     * @return OSイメージID
     */
    @Query(name = "ImageId", require = true)
    public String getImageId() {
        return imageId;
    }

    /**
     * OSイメージIDを設定します。
     *
     * @param imageId
     *            OSイメージID
     */
    public void setImageId(String imageId) {
        this.imageId = imageId;
    }

    /**
     * OSイメージIDを設定し、自オブジェクトを返します。
     *
     * @param imageId
     *            OSイメージID
     * @return 自オブジェクト
     */
    public NiftyAssociateImageRequest withImageId(String imageId) {
        setImageId(imageId);
        return this;
    }

    /**
     * 公開レベルを取得します。
     *
     * @return 公開レベル
     */
    @Query(name = "IsPublic", require = true)
    public boolean getIsPublic() {
        return isPublic;
    }

    /**
     * 公開レベルを設定します。
     *
     * @param isPublic
     *            公開レベル
     */
    public void setIsPublic(boolean isPublic) {
        this.isPublic = isPublic;
    }

    /**
     * 公開レベルを設定し、自オブジェクトを返します。
     *
     * @param isPublic
     *            公開レベル
     * 
     * @return 自オブジェクト
     */
    public NiftyAssociateImageRequest withIsPublic(boolean isPublic) {
        setIsPublic(isPublic);
        return this;
    }

    /**
     * 再配布設定を取得します。
     *
     * @return 再配布設定
     */
    @Query(name = "IsRedistribute")
    public boolean getIsRedistribute() {
        return isRedistribute;
    }

    /**
     * 再配布設定を設定します。
     *
     * @param isRedistribute
     *            再配布設定
     */
    public void setIsRedistribute(boolean isRedistribute) {
        this.isRedistribute = isRedistribute;
    }

    /**
     * 再配布設定許可を設定し、自オブジェクトを返します。
     *
     * @param isRedistribute
     *            再配布設定許可
     * @return 自オブジェクト
     */
    public NiftyAssociateImageRequest withRedistribute(boolean isRedistribute) {
        setIsPublic(isRedistribute);
        return this;
    }

    /**
     * 配布先@nifty 法人IDリストを取得します。
     *
     * @return 配布先@nifty 法人IDリスト
     */
    @Query(name = "DistributionId")
    public List<String> getDistributionIds() {
        return distributionIds;
    }

    /**
     * 配布先@nifty 法人IDリストを設定します。
     *
     * @param distributionIds
     *            配布先@nifty 法人IDリスト
     */
    public void setDistributionIds(List<String> distributionIds) {
        this.distributionIds = distributionIds;
    }

    /**
     * 配布先@nifty法人IDの配列を設定し、自オブジェクトを返します。
     *
     * @param distributionIds
     *            配布先@nifty法人IDリスト
     * @return 自オブジェクト
     */
    public NiftyAssociateImageRequest withDistributionIds(String... distributionIds) {
        if (this.distributionIds == null)
            this.distributionIds = new ArrayList<String>();
        for (String distributionId : distributionIds) {
            getDistributionIds().add(distributionId);
        }
        return this;
    }

    /**
     * 配布先@nifty法人IDリストを設定し、自オブジェクトを返します。
     * 
     * @param distributionIds
     *            配布先@nifty法人IDリスト
     * @return 自オブジェクト
     */
    public NiftyAssociateImageRequest withDistributionIds(Collection<String> distributionIds) {
        if (this.distributionIds == null)
            this.distributionIds = new ArrayList<String>();
        if (distributionIds != null) {
            getDistributionIds().addAll(distributionIds);
        }
        return this;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("[imageId=");
        builder.append(imageId);
        builder.append(", isPublic=");
        builder.append(isPublic);
        builder.append(", isRedistribute=");
        builder.append(isRedistribute);
        builder.append(", distributionId=");
        builder.append(distributionIds);
        builder.append("]");
        return builder.toString();
    }
}
