/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.firewall.model;

import java.util.ArrayList;
import java.util.List;

import com.nifty.cloud.sdk.firewall.model.transform.InstancesConverter;
import com.nifty.cloud.sdk.firewall.model.transform.InstanceNetworkInterfaceConverter;
import com.nifty.cloud.sdk.firewall.model.transform.IpPermissionConverter;
import com.nifty.cloud.sdk.firewall.model.transform.RouterConverter;
import com.nifty.cloud.sdk.firewall.model.transform.RouterNetworkInterfaceConverter;
import com.nifty.cloud.sdk.firewall.model.transform.VpnGatewayConverter;
import com.nifty.cloud.sdk.firewall.model.transform.VpnGatewayNetworkInterfaceConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 *
 * ファイアウォールグループ情報クラス。<br />
 * このクラスはファイアウォールグループ情報を格納します。
 *
 */
public class SecurityGroup {

	@XStreamAlias("ownerId")
	private String ownerId;

	/** ファイアウォールグループ名 */
	@XStreamAlias("groupName")
	private String groupName;

	/** ファイアウォールグループのメモ */
	@XStreamAlias("groupDescription")
	private String groupDescription;

	/** ファイアウォールグループの処理ステータス */
	@XStreamAlias("groupStatus")
	private String groupStatus;

	/** 許可ルール情報リスト */
	@XStreamAlias("ipPermissions")
	@XStreamConverter(IpPermissionConverter.class)
	private List<IpPermission> ipPermissions;

	/** 適用しているサーバー情報リスト */
	@XStreamAlias("instancesSet")
	@XStreamConverter(InstancesConverter.class)
	private List<String> instanceIds;

	/** ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報リスト */
	@XStreamAlias("instancesNetworkInterfaceSet")
	@XStreamConverter(InstanceNetworkInterfaceConverter.class)
	private List<InstanceNetworkInterface> instanceNetworkInterfaces;

	/** VM全体にファイアウォール適用しているルーター情報リスト */
	@XStreamAlias("routerSet")
	@XStreamConverter(RouterConverter.class)
	private List<Router> routers;

	/** ネットワークインターフェイスごとにファイアウォール適用しているルーター情報リスト */
	@XStreamAlias("routerNetworkInterfaceSet")
	@XStreamConverter(RouterNetworkInterfaceConverter.class)
	private List<RouterNetworkInterface> routerNetworkInterfaces;

	/** VM全体にファイアウォール適用しているVPNゲートウェイ情報リスト */
	@XStreamAlias("vpnGatewaySet")
	@XStreamConverter(VpnGatewayConverter.class)
	private List<VpnGateway> vpnGateways;

	/** ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報リスト */
	@XStreamAlias("vpnGatewayNetworkInterfaceSet")
	@XStreamConverter(VpnGatewayNetworkInterfaceConverter.class)
	private List<VpnGatewayNetworkInterface> vpnGatewayNetworkInterfaces;

	/** ルール数上限 */
	private Integer groupRuleLimit;

	/** ログ取得件数 */
	private Integer groupLogLimit;

	/** Windows サーバー BroadCast 通信ログの抑止状態 */
	private Boolean groupLogFilterNetBios;

	/** ゾーン名 */
	private String availabilityZone;

	/**
	 * ownerIdを取得します。
	 * @return ownerId
	 */
	public String getOwnerId() {
		return ownerId;
	}

	/**
	 * ownerIdを設定します。
	 * @param ownerId
	 */
	public void setOwnerId(String ownerId) {
		this.ownerId = ownerId;
	}

	/**
	 * ownerIdを設定し、自オブジェクトを返します。
	 * @param ownerId
	 * @return 自オブジェクト
	 */
	public SecurityGroup withOwnerId(String ownerId) {
		this.ownerId = ownerId;
		return this;
	}

	/**
	 * ファイアウォールグループ名を取得します。
	 *
	 * @return ファイアウォールグループ名
	 */
	public String getGroupName() {
		return groupName;
	}

	/**
	 * ファイアウォールグループ名を設定します。
	 *
	 * @param groupName ファイアウォールグループ名
	 */
	public void setGroupName(String groupName) {
		this.groupName = groupName;
	}

	/**
	 * ファイアウォールグループ名を設定し、自オブジェクトを返します。
	 *
	 * @param groupName ファイアウォールグループ名
	 * @return 自オブジェクト
	 */
	public SecurityGroup withGroupName(String groupName) {
		setGroupName(groupName);
		return this;
	}

	/**
	 * ファイアウォールグループのメモを取得します。
	 *
	 * @return ファイアウォールグループのメモ
	 */
	public String getGroupDescription() {
		return groupDescription;
	}

	/**
	 * ファイアウォールグループのメモを設定します。
	 *
	 * @param groupDescription ファイアウォールグループのメモ
	 */
	public void setGroupDescription(String groupDescription) {
		this.groupDescription = groupDescription;
	}

	/**
	 * ファイアウォールグループのメモを設定し、自オブジェクトを返します。
	 *
	 * @param groupDescription ファイアウォールグループのメモ
	 * @return 自オブジェクト
	 */
	public SecurityGroup withGroupDescription(String groupDescription) {
		setGroupDescription(groupDescription);
		return this;
	}

	/**
	 * ファイアウォールグループの処理ステータスを取得します。
	 * @return groupStatus ファイアウォールグループの処理ステータス
	 */
	public String getGroupStatus() {
		return groupStatus;
	}

	/**
	 * ファイアウォールグループの処理ステータスを設定します。
	 * @param groupStatus ファイアウォールグループの処理ステータス
	 */
	public void setGroupStatus(String groupStatus) {
		this.groupStatus = groupStatus;
	}

	/**
	 * フファイアウォールグループの処理ステータスを設定し、自オブジェクトを返します。
	 *
	 * @param groupStatus ファイアウォールグループの処理ステータス
	 * @return 自オブジェクト
	 */
	public SecurityGroup withGroupStatus(String groupStatus) {
		setGroupStatus(groupStatus);
		return this;
	}

	/**
	 * 許可ルール情報リストを取得します。
	 *
	 * @return 許可ルール情報リスト
	 */
	public List<IpPermission> getIpPermissions() {
		return ipPermissions;
	}

	/**
	 * 許可ルール情報リストを設定します。
	 *
	 * @param ipPermissions 許可ルール情報リスト
	 */
	public void setIpPermissions(List<IpPermission> ipPermissions) {
		this.ipPermissions = ipPermissions;
	}

	/**
	 * 許可ルール情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param ipPermissions 許可ルール情報の配列
	 * @return 自オブジェクト
	 */
	public SecurityGroup withIpPermissions(IpPermission... ipPermissions) {
		if (this.ipPermissions == null)
			this.ipPermissions = new ArrayList<IpPermission>();
		for (IpPermission ipPermission : ipPermissions) {
			getIpPermissions().add(ipPermission);
		}
		return this;
	}

	/**
	 * 許可ルール情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param ipPermissions 許可ルール情報リスト
	 * @return 自オブジェクト
	 */
	public SecurityGroup withIpPermissions(List<IpPermission> ipPermissions) {
		if (this.ipPermissions == null)
			this.ipPermissions = new ArrayList<IpPermission>();
		if (ipPermissions != null) {
			getIpPermissions().addAll(ipPermissions);
		}
		return this;
	}

	/**
	 * 適用しているサーバー情報リストを取得します。
	 *
	 * @return 適用しているサーバー情報リスト
	 */
	public List<String> getInstanceIds() {
		return instanceIds;
	}

	/**
	 * 適用しているサーバー情報リストを設定します。
	 *
	 * @param instanceIds 適用しているサーバー情報リスト
	 */
	public void setInstanceIds(List<String> instanceIds) {
		this.instanceIds = instanceIds;
	}

	/**
	 * 適用しているサーバー情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param instanceIds 適用しているサーバー情報の配列
	 * @return 自オブジェクト
	 */
	public SecurityGroup withInstanceIds(String... instanceIds) {
		if (this.instanceIds == null)
			this.instanceIds = new ArrayList<String>();
		for (String instanceId : instanceIds) {
			getInstanceIds().add(instanceId);
		}
		return this;
	}

	/**
	 * 適用しているサーバー情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param instanceIds 適用しているサーバー情報リスト
	 * @return 自オブジェクト
	 */
	public SecurityGroup withInstanceIds(List<String> instanceIds) {
		if (this.instanceIds == null)
			this.instanceIds = new ArrayList<String>();
		if (instanceIds != null) {
			getInstanceIds().addAll(instanceIds);
		}
		return this;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報リストを取得します。
	 *
	 * @return ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報リスト
	 */
	public List<InstanceNetworkInterface> getInstanceNetworkInterfaces() {
		return instanceNetworkInterfaces;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報リストを設定します。
	 *
	 * @param instanceNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報リスト
	 */
	public void setInstanceNetworkInterfaces(List<InstanceNetworkInterface> instanceNetworkInterfaces) {
		this.instanceNetworkInterfaces = instanceNetworkInterfaces;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param instanceNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報の配列
	 * @return 自オブジェクト
	 */
	public SecurityGroup withInstanceNetworkInterfaces(InstanceNetworkInterface... instanceNetworkInterfaces) {
		if (this.instanceNetworkInterfaces == null)
			this.instanceNetworkInterfaces = new ArrayList<InstanceNetworkInterface>();
		for (InstanceNetworkInterface instanceNetworkInterface : instanceNetworkInterfaces) {
			getInstanceNetworkInterfaces().add(instanceNetworkInterface);
		}
		return this;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param instanceNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているサーバー情報リスト
	 * @return 自オブジェクト
	 */
	public SecurityGroup withInstanceNetworkInterfaces(List<InstanceNetworkInterface> instanceNetworkInterfaces) {
		if (this.instanceNetworkInterfaces == null)
			this.instanceNetworkInterfaces = new ArrayList<InstanceNetworkInterface>();
		if (instanceNetworkInterfaces != null) {
			getInstanceNetworkInterfaces().addAll(instanceNetworkInterfaces);
		}
		return this;
	}

	/**
	 * VM全体にファイアウォール適用しているルーター情報リストを取得します。
	 *
	 * @return VM全体にファイアウォール適用しているルーター情報
	 */
	public List<Router> getRouters() {
		return routers;
	}

	/**
	 * VM全体にファイアウォール適用しているルーター情報リストを設定します。
	 *
	 * @param routers VM全体にファイアウォール適用しているルーター情報リスト
	 */
	public void setRouters(List<Router> routers) {
		this.routers = routers;
	}

	/**
	 * VM全体にファイアウォール適用しているルーター情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param routers VM全体にファイアウォール適用しているルーター情報の配列
	 * @return 自オブジェクト
	 */
	public SecurityGroup withRouters(Router... routers) {
		if (this.routers == null)
			this.routers = new ArrayList<Router>();
		for (Router router : routers) {
			getRouters().add(router);
		}
		return this;
	}

	/**
	 * VM全体にファイアウォール適用しているルーター情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param routers VM全体にファイアウォール適用しているルーター情報リスト
	 * @return 自オブジェクト
	 */
	public SecurityGroup withRouters(List<Router> routers) {
		if (this.routers == null)
			this.routers = new ArrayList<Router>();
		if (routers != null) {
			getRouters().addAll(routers);
		}
		return this;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているルーター情報リストを取得します。
	 *
	 * @return ネットワークインターフェイスごとにファイアウォール適用しているルーター情報リスト
	 */
	public List<RouterNetworkInterface> getRouterNetworkInterfaces() {
		return routerNetworkInterfaces;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているルーター情報リストを設定します。
	 *
	 * @param routerNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているルーター情報リスト
	 */
	public void setRouterNetworkInterfaces(List<RouterNetworkInterface> routerNetworkInterfaces) {
		this.routerNetworkInterfaces = routerNetworkInterfaces;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているルーター情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param routerNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているルーター情報の配列
	 * @return 自オブジェクト
	 */
	public SecurityGroup withRouterNetworkInterfaces(RouterNetworkInterface... routerNetworkInterfaces) {
		if (this.routerNetworkInterfaces == null)
			this.routerNetworkInterfaces = new ArrayList<RouterNetworkInterface>();
		for (RouterNetworkInterface routerNetworkInterface : routerNetworkInterfaces) {
			getRouterNetworkInterfaces().add(routerNetworkInterface);
		}
		return this;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているルーター情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param routerNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているルーター情報リスト
	 * @return 自オブジェクト
	 */
	public SecurityGroup withRouterNetworkInterfaces(List<RouterNetworkInterface> routerNetworkInterfaces) {
		if (this.routerNetworkInterfaces == null)
			this.routerNetworkInterfaces = new ArrayList<RouterNetworkInterface>();
		if (routerNetworkInterfaces != null) {
			getRouterNetworkInterfaces().addAll(routerNetworkInterfaces);
		}
		return this;
	}

	/**
	 * VM全体にファイアウォール適用しているVPNゲートウェイ情報リストを取得します。
	 *
	 * @return VM全体にファイアウォール適用しているVPNゲートウェイ情報リスト
	 */
	public List<VpnGateway> getVpnGateways() {
		return vpnGateways;
	}

	/**
	 * VM全体にファイアウォール適用しているVPNゲートウェイ情報リストを設定します。
	 *
	 * @param vpnGateways VM全体にファイアウォール適用しているVPNゲートウェイ情報リスト
	 */
	public void setVpnGateways(List<VpnGateway> vpnGateways) {
		this.vpnGateways = vpnGateways;
	}

	/**
	 * VM全体にファイアウォール適用しているVPNゲートウェイ情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param vpnGateways VM全体にファイアウォール適用しているVPNゲートウェイ情報の配列
	 * @return 自オブジェクト
	 */
	public SecurityGroup withVpnGateways(VpnGateway... vpnGateways) {
		if (this.vpnGateways == null)
			this.vpnGateways = new ArrayList<VpnGateway>();
		for (VpnGateway vpnGateway : vpnGateways) {
			getVpnGateways().add(vpnGateway);
		}
		return this;
	}

	/**
	 * VM全体にファイアウォール適用しているVPNゲートウェイ情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param vpnGateways VM全体にファイアウォール適用しているVPNゲートウェイ情報リスト
	 * @return 自オブジェクト
	 */
	public SecurityGroup withVpnGateways(List<VpnGateway> vpnGateways) {
		if (this.vpnGateways == null)
			this.vpnGateways = new ArrayList<VpnGateway>();
		if (vpnGateways != null) {
			getVpnGateways().addAll(vpnGateways);
		}
		return this;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報リストを取得します。
	 *
	 * @return ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報リスト
	 */
	public List<VpnGatewayNetworkInterface> getVpnGatewayNetworkInterfaces() {
		return vpnGatewayNetworkInterfaces;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報リストを設定します。
	 *
	 * @param vpnGatewayNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報リスト
	 */
	public void setVpnGatewayNetworkInterfaces(List<VpnGatewayNetworkInterface> vpnGatewayNetworkInterfaces) {
		this.vpnGatewayNetworkInterfaces = vpnGatewayNetworkInterfaces;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param vpnGatewayNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報の配列
	 * @return 自オブジェクト
	 */
	public SecurityGroup withVpnGatewayNetworkInterfaces(VpnGatewayNetworkInterface... vpnGatewayNetworkInterfaces) {
		if (this.vpnGatewayNetworkInterfaces == null)
			this.vpnGatewayNetworkInterfaces = new ArrayList<VpnGatewayNetworkInterface>();
		for (VpnGatewayNetworkInterface vpnGatewayNetworkInterface : vpnGatewayNetworkInterfaces) {
			getVpnGatewayNetworkInterfaces().add(vpnGatewayNetworkInterface);
		}
		return this;
	}

	/**
	 * ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param vpnGatewayNetworkInterfaces ネットワークインターフェイスごとにファイアウォール適用しているVPNゲートウェイ情報リスト
	 * @return 自オブジェクト
	 */
	public SecurityGroup withVpnGatewayNetworkInterfaces(List<VpnGatewayNetworkInterface> vpnGatewayNetworkInterfaces) {
		if (this.vpnGatewayNetworkInterfaces == null)
			this.vpnGatewayNetworkInterfaces = new ArrayList<VpnGatewayNetworkInterface>();
		if (vpnGatewayNetworkInterfaces != null) {
			getVpnGatewayNetworkInterfaces().addAll(vpnGatewayNetworkInterfaces);
		}
		return this;
	}


	/**
	 * ルール数上限を取得します。
	 *
	 * @return ルール数上限
	 */
	public Integer getGroupRuleLimit() {
		return groupRuleLimit;
	}

	/**
	 * ルール数上限を設定します。
	 *
	 * @param groupRuleLimit ルール数上限
	 */
	public void setGroupRuleLimit(Integer groupRuleLimit) {
		this.groupRuleLimit = groupRuleLimit;
	}

	/**
	 * ルール数上限を設定し、自オブジェクトを返します。
	 * @param groupRuleLimit ルール数上限
	 * @return 自オブジェクト
	 */
	public SecurityGroup withGroupRuleLimit(Integer groupRuleLimit) {
		setGroupRuleLimit(groupRuleLimit);
		return this;
	}

	/**
	 * ログ取得件数を取得します。
	 *
	 * @return ログ取得件数
	 */
	public Integer getGroupLogLimit() {
		return groupLogLimit;
	}

	/**
	 * ログ取得件数を設定します。
	 *
	 * @param groupLogLimit ログ取得件数
	 */
	public void setGroupLogLimit(Integer groupLogLimit) {
		this.groupLogLimit = groupLogLimit;
	}

	/**
	 * ログ取得件数を設定し、自オブジェクトを返します。
	 *
	 * @param groupLogLimit ログ取得件数
	 * @return 自オブジェクト
	 */
	public SecurityGroup withGroupLogLimit(Integer groupLogLimit) {
		setGroupLogLimit(groupLogLimit);
		return this;
	}


	/**
	 * WindowsサーバーBroadcastログの抑止設定情報を取得します
	 * @return boolean true:設定する  false:設定しない
	 */
	public Boolean isGroupLogFilterNetBios() {
		return groupLogFilterNetBios;
	}

	/**
	 * WindowsサーバーBroadcastログの抑止設定を登録します
	 * @param groupLogFilterNetBios  (true:設定する  false:設定しない)
	 */
	public void setGroupLogFilterNetBios(boolean groupLogFilterNetBios) {
		this.groupLogFilterNetBios = groupLogFilterNetBios;
	}

	/**
	 * WindowsサーバーBroadcastログの抑止設定を登録し、自オブジェクトを返します
	 * @param groupLogFilterNetBios (true:設定する  false:設定しない)
	 * @return 自オブジェクト
	 */
	public SecurityGroup withGroupLogFilterNetBios(boolean groupLogFilterNetBios) {
		setGroupLogFilterNetBios(groupLogFilterNetBios);
		return this;
	}


	/**
	 * ゾーン名を取得します。
	 * @return ゾーン名
	 */
	public String getAvailabilityZone() {
		return availabilityZone;
	}

	/**
	 * ゾーン名を設定します。
	 * @param availabilityZone ゾーン名
	 */
	public void setAvailabilityZone(String availabilityZone) {
		this.availabilityZone = availabilityZone;
	}

	/**
	 * ゾーン名を設定し、自オブジェクトを返します。
	 * @param availabilityZone ゾーン名
	 * @return 自オブジェクト
	 */


	public SecurityGroup withAvailabilityZone(String availabilityZone) {
		setAvailabilityZone(availabilityZone);
		return this;
	}



	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[ownerId=");
		builder.append(ownerId);
		builder.append(", groupName=");
		builder.append(groupName);
		builder.append(", groupDescription=");
		builder.append(groupDescription);
		builder.append(", groupStatus=");
		builder.append(groupStatus);
		builder.append(", ipPermissions=");
		builder.append(ipPermissions);
		builder.append(", instanceIds=");
		builder.append(instanceIds);
		builder.append(", instanceNetworkInterfaces=");
		builder.append(instanceNetworkInterfaces);
		builder.append(", routers=");
		builder.append(routers);
		builder.append(", routerNetworkInterfaces=");
		builder.append(routerNetworkInterfaces);
		builder.append(", vpnGateways=");
		builder.append(vpnGateways);
		builder.append(", vpnGatewayNetworkInterfaces=");
		builder.append(vpnGatewayNetworkInterfaces);
		builder.append(", groupRuleLimit=");
		builder.append(groupRuleLimit);
		builder.append(", groupLogLimit=");
		builder.append(groupLogLimit);
		builder.append(" ,groupLogFilterNetBios=");
		builder.append(groupLogFilterNetBios);
		builder.append(" ,availabilityZone=");
		builder.append(availabilityZone);
		builder.append("]");
		return builder.toString();
	}
}
