/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.firewall.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Request;
import com.nifty.cloud.sdk.annotation.Action;
import com.nifty.cloud.sdk.annotation.Query;

/**
 * NiftyRegisterRoutersWithSecurityGroupリクエストクラス。<br />
 * このクラスはNiftyRegisterRoutersWithSecurityGroupへのリクエストを構築します。
 *
 */
@Action("NiftyRegisterRoutersWithSecurityGroup")
public class NiftyRegisterRoutersWithSecurityGroupRequest implements Request {

	/** ファイアウォールグループ名 */
	private String groupName;

	/** ルーター情報 */
	private List<RouterIdSet> routers;

	/**
	 * デフォルトコンストラクタ。
	 */
	public NiftyRegisterRoutersWithSecurityGroupRequest() {
	}

	/**
	 * ファイアウォールグループ名,ルーター情報を指定し、NiftyRegisterRoutersWithSecurityGroupRequestを構築します。
	 *
	 * @param groupName ファイアウォールグループ名
	 * @param routers ルーター情報
	 */
	public NiftyRegisterRoutersWithSecurityGroupRequest(String groupName, List<RouterIdSet> routers) {
		this.groupName = groupName;
		this.routers = routers;
	}

	/**
	 * ファイアウォールグループ名を取得します。
	 *
	 * @return ファイアウォールグループ名
	 */
	@Query(name="GroupName", require=true)
	public String getGroupName() {
		return groupName;
	}

	/**
	 * ファイアウォールグループ名を設定します。
	 *
	 * @param groupName ファイアウォールグループ名
	 */
	public void setGroupName(String groupName) {
		this.groupName = groupName;
	}

	/**
	 * ファイアウォールグループ名を設定し、自オブジェクトを返します。
	 *
	 * @param groupName ファイアウォールグループ名
	 * @return 自オブジェクト
	 */
	public NiftyRegisterRoutersWithSecurityGroupRequest withGroupName(String groupName) {
		setGroupName(groupName);
		return this;
	}

	/**
	 * ルーター情報を取得します。
	 *
	 * @return ルーター情報
	 */
	@Query(name="RouterSet", require=true)
	public List<RouterIdSet> getRouters() {
		return routers;
	}

	/**
	 * ルーター情報を設定します。
	 *
	 * @param routers ルーター情報
	 */
	public void setRouters(List<RouterIdSet> routers) {
		this.routers = routers;
	}

	/**
	 * ルーター情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param routers ルーター情報の配列
	 * @return 自オブジェクト
	 */
	public NiftyRegisterRoutersWithSecurityGroupRequest withRouters(RouterIdSet ... routers) {
		if (this.routers == null) this.routers = new ArrayList<RouterIdSet>();
		for (RouterIdSet router : routers) {
			getRouters().add(router);
		}
		return this;
	}

	/**
	 * ルーター情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param routers ルーター情報リスト
	 * @return 自オブジェクト
	 */
	public NiftyRegisterRoutersWithSecurityGroupRequest withRouters(Collection<RouterIdSet> routers) {
		if (this.routers == null) this.routers = new ArrayList<RouterIdSet>();
		if (routers != null) {
			getRouters().addAll(routers);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[groupName=");
		builder.append(groupName);
		builder.append(", routers=");
		builder.append(routers);
		builder.append("]");
		return builder.toString();
	}
}
