/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model.transform;

import java.util.ArrayList;
import java.util.List;

import com.nifty.cloud.sdk.dhcp.model.DhcpConfiguration;
import com.nifty.cloud.sdk.dhcp.model.DhcpOptions;
import com.nifty.cloud.sdk.dhcp.model.Tag;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * DhcpOptionsType変換クラス。<br />
 * このクラスはレスポンスXMLのdhcpOptionsSet要素からDhcpOptionsクラスのリストを生成します。
 */
public class DhcpOptionsSetConverter implements Converter {

	private static final String ELEMENT_DHCP_OPTIONS_ID = "dhcpOptionsId";
	private static final String ELEMENT_DHCP_CONFIGURATION_SET = "dhcpConfigurationSet";
	private static final String ELEMENT_TAG_SET = "tagSet";

	/**
	 * 変換可能なクラスか判定します。
	 *
	 * @param clazz クラス
	 * @return 変換可能な場合true, 変換不可能な場合false
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public boolean canConvert(Class clazz) {
		Class[] interfaces = clazz.getInterfaces();
		for (Class i : interfaces) {
			if (i == List.class) {
				return true;
			}
		}
		return false;
	}

	/**
	 * {@literal
	 * dhcpOptionsSetをList<DhcpOptions>クラスに変換します。}
	 *
	 * @param reader XML読込リーダ
	 * @param context
	 * @return {@literal List<DhcpOptions>オブジェクト}
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader, UnmarshallingContext)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
		List<DhcpOptions> dhcpOptionsSet = new ArrayList<DhcpOptions>();
		Converter dhcpConfigurationSetConverter = new DhcpConfigurationSetConverter();
		Converter tagSetConverter = new TagSetConverter();

		String name = null;
		while (reader.hasMoreChildren()) {
			reader.moveDown();
			DhcpOptions dhcpOptions = new DhcpOptions();

			while (reader.hasMoreChildren()) {
				reader.moveDown();

				name = reader.getNodeName();
				if (name.equals(ELEMENT_DHCP_OPTIONS_ID)) {
					dhcpOptions.setDhcpOptionsId(reader.getValue());
				} else if (name.equals(ELEMENT_DHCP_CONFIGURATION_SET)) {
					dhcpOptions.setDhcpConfigurations((List<DhcpConfiguration>) dhcpConfigurationSetConverter.unmarshal(reader, context));
				} else if (name.equals(ELEMENT_TAG_SET)) {
					dhcpOptions.setTagSet((List<Tag>) tagSetConverter.unmarshal(reader, context));
				}

				reader.moveUp();
			}

			dhcpOptionsSet.add(dhcpOptions);
			reader.moveUp();
		}

		return dhcpOptionsSet;
	}

	/**
	 * 未実装メソッド
	 */
	@Override
	public void marshal(Object paramObject,
			HierarchicalStreamWriter paramHierarchicalStreamWriter,
			MarshallingContext paramMarshallingContext) {
	}
}
