/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model.transform;

import java.util.ArrayList;
import java.util.List;

import com.nifty.cloud.sdk.dhcp.model.DhcpIpAddress;
import com.nifty.cloud.sdk.tools.DateUtils;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * DhcpIpAddressSetType変換クラス。<br />
 * このクラスはレスポンスXMLのdhcpIpAddressSet要素からDhcpIpAddressクラスのリストを生成します。
 */
public class DhcpIpAddressSetConverter implements Converter {

	private static final String ELEMENT_IP_ADDRESS = "ipAddress";
	private static final String ELEMENT_MAC_ADDRESS = "macAddress";
	private static final String ELEMENT_LEASE_TYPE = "leaseType";
	private static final String ELEMENT_LEASE_EXPIRATION = "leaseExpiration";
	private static final String ELEMENT_CLIENT_NAME = "clientName";
	private static final String ELEMENT_DESCRIPTION = "description";

	/**
	 * 変換可能なクラスか判定します。
	 *
	 * @param clazz クラス
	 * @return 変換可能な場合true, 変換不可能な場合false
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public boolean canConvert(Class clazz) {
		Class[] interfaces = clazz.getInterfaces();
		for (Class i : interfaces) {
			if (i == List.class) {
				return true;
			}
		}
		return false;
	}

	/**
	 * {@literal
	 * dhcpIpAddressSetをList<DhcpIpAddress>クラスに変換します。}
	 *
	 * @param reader XML読込リーダ
	 * @param context
	 * @return {@literal List<DhcpIpAddress>オブジェクト}
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader, UnmarshallingContext)
	 */
	@Override
	public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
		List<DhcpIpAddress> dhcpIpAddresses = new ArrayList<DhcpIpAddress>();

		String name = null;
		while (reader.hasMoreChildren()) {
			reader.moveDown();
			DhcpIpAddress dhcpIpAddress = new DhcpIpAddress();

			while (reader.hasMoreChildren()) {
				reader.moveDown();

				name = reader.getNodeName();
				if (name.equals(ELEMENT_IP_ADDRESS)) {
					dhcpIpAddress.setIpAddress(reader.getValue());
				} else if (name.equals(ELEMENT_MAC_ADDRESS)) {
					dhcpIpAddress.setMacAddress(reader.getValue());
				} else if (name.equals(ELEMENT_LEASE_TYPE)) {
					dhcpIpAddress.setLeaseType(reader.getValue());
				} else if (name.equals(ELEMENT_LEASE_EXPIRATION)) {
					dhcpIpAddress.setLeaseExpiration(DateUtils.toDate(reader.getValue()));
				} else if (name.equals(ELEMENT_CLIENT_NAME)) {
					dhcpIpAddress.setClientName(reader.getValue());
				} else if (name.equals(ELEMENT_DESCRIPTION)) {
					dhcpIpAddress.setDescription(reader.getValue());
				}

				reader.moveUp();
			}

			dhcpIpAddresses.add(dhcpIpAddress);
			reader.moveUp();
		}

		return dhcpIpAddresses;
	}

	/**
	 * 未実装メソッド
	 */
	@Override
	public void marshal(Object paramObject,
			HierarchicalStreamWriter paramHierarchicalStreamWriter,
			MarshallingContext paramMarshallingContext) {
	}
}
