/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model.transform;

import java.util.List;

import com.nifty.cloud.sdk.dhcp.model.DhcpIpAddress;
import com.nifty.cloud.sdk.dhcp.model.DhcpIpAddressInformation;
import com.nifty.cloud.sdk.dhcp.model.IpAddressPool;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * DhcpIpAddressInformationType変換クラス。<br />
 * このクラスはレスポンスXMLのdhcpIpAddressInformation要素からDhcpIpAddressInformationクラスを生成します。
 */
public class DhcpIpAddressInformationConverter implements Converter {

	private static final String ELEMENT_IP_ADDRESS_POOL_SET = "ipAddressPoolSet";
	private static final String ELEMENT_DHCP_IP_ADDRESS_SET = "dhcpIpAddressSet";

	/**
	 * 変換可能なクラスか判定します。
	 *
	 * @param clazz クラス
	 * @return 変換可能な場合true, 変換不可能な場合false
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public boolean canConvert(Class clazz) {
		return clazz == DhcpIpAddressInformation.class;
	}

	/**
	 * {@literal
	 * dhcpIpAddressInformationをDhcpIpAddressInformationクラスに変換します。}
	 *
	 * @param reader XML読込リーダ
	 * @param context
	 * @return {@literal DhcpIpAddressInformationオブジェクト}
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader, UnmarshallingContext)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
		DhcpIpAddressInformation dhcpIpAddressInformation = new DhcpIpAddressInformation();
		Converter ipAddressPoolSetConverter = new IpAddressPoolSetConverter();
		Converter dhcpIpAddressSetConverter = new DhcpIpAddressSetConverter();

		String name = null;
		while (reader.hasMoreChildren()) {
			reader.moveDown();

			name = reader.getNodeName();
			if (name.equals(ELEMENT_IP_ADDRESS_POOL_SET)) {
				dhcpIpAddressInformation.setIpAddressPools((List<IpAddressPool>) ipAddressPoolSetConverter.unmarshal(reader, context));
			} else if (name.equals(ELEMENT_DHCP_IP_ADDRESS_SET)) {
				dhcpIpAddressInformation.setDhcpIpAddresses((List<DhcpIpAddress>) dhcpIpAddressSetConverter.unmarshal(reader, context));
			}

			reader.moveUp();
		}

		return dhcpIpAddressInformation;
	}

	/**
	 * 未実装メソッド
	 */
	@Override
	public void marshal(Object paramObject,
			HierarchicalStreamWriter paramHierarchicalStreamWriter,
			MarshallingContext paramMarshallingContext) {
	}
}
