/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model.transform;

import java.util.ArrayList;
import java.util.List;

import com.nifty.cloud.sdk.dhcp.model.DhcpConfiguration;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * DhcpConfigurationSetType変換クラス。<br />
 * このクラスはレスポンスXMLのdhcpConfigurationSet要素からDhcpConfigurationクラスのリストを生成します。
 */
public class DhcpConfigurationSetConverter implements Converter {

	private static final String ELEMENT_KEY = "key";
	private static final String ELEMENT_VALUE_SET = "valueSet";

	/**
	 * 変換可能なクラスか判定します。
	 *
	 * @param clazz クラス
	 * @return 変換可能な場合true, 変換不可能な場合false
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public boolean canConvert(Class clazz) {
		Class[] interfaces = clazz.getInterfaces();
		for (Class i : interfaces) {
			if (i == List.class) {
				return true;
			}
		}
		return false;
	}

	/**
	 * {@literal
	 * dhcpConfigurationSetをList<DhcpConfiguration>クラスに変換します。}
	 *
	 * @param reader XML読込リーダ
	 * @param context
	 * @return {@literal List<DhcpConfiguration>オブジェクト}
	 *
	 * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader, UnmarshallingContext)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
		List<DhcpConfiguration> dhcpConfigurations = new ArrayList<DhcpConfiguration>();
		Converter valueSetConverter = new ValueSetConverter();

		String name = null;
		while (reader.hasMoreChildren()) {
			reader.moveDown();
			DhcpConfiguration dhcpConfiguration = new DhcpConfiguration();

			while (reader.hasMoreChildren()) {
				reader.moveDown();

				name = reader.getNodeName();
				if (name.equals(ELEMENT_KEY)) {
					dhcpConfiguration.setKey(reader.getValue());
				} else if (name.equals(ELEMENT_VALUE_SET)) {
					dhcpConfiguration.setValues((List<String>) valueSetConverter.unmarshal(reader, context));
				}

				reader.moveUp();
			}

			dhcpConfigurations.add(dhcpConfiguration);
			reader.moveUp();
		}

		return dhcpConfigurations;
	}

	/**
	 * 未実装メソッド
	 */
	@Override
	public void marshal(Object paramObject,
			HierarchicalStreamWriter paramHierarchicalStreamWriter,
			MarshallingContext paramMarshallingContext) {
	}
}
