/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Result;
import com.nifty.cloud.sdk.dhcp.model.transform.NiftyDescribeDhcpConfigsResultConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * NiftyDescribeDhcpConfigs処理結果クラス。<br />
 * このクラスはNiftyDescribeDhcpConfigsのレスポンスを格納します。
 */
@XStreamAlias("NiftyDescribeDhcpConfigsResponse")
@XStreamConverter(NiftyDescribeDhcpConfigsResultConverter.class)
public class NiftyDescribeDhcpConfigsResult extends Result {

	/** リクエスト識別子 */
	private String requestId;

	/** DHCPコンフィグ情報リスト */
	private List<DhcpConfig> dhcpConfigs;

	/**
	 * リクエスト識別子を取得します。
	 *
	 * @return リクエスト識別子
	 */
	public String getRequestId() {
		return requestId;
	}

	/**
	 * リクエスト識別子を設定します。
	 *
	 * @param requestId リクエスト識別子
	 */
	public void setRequestId(String requestId) {
		this.requestId = requestId;
	}

	/**
	 * リクエスト識別子を設定し、自オブジェクトを返します。
	 *
	 * @param requestId リクエスト識別子
	 * @return 自オブジェクト
	 */
	public NiftyDescribeDhcpConfigsResult withRequestId(String requestId) {
		setRequestId(requestId);
		return this;
	}

	/**
	 * DHCPコンフィグ情報リストを取得します。
	 *
	 * @return DHCPコンフィグ情報リスト
	 */
	public List<DhcpConfig> getDhcpConfigs() {
		return dhcpConfigs;
	}

	/**
	 * DHCPコンフィグ情報リストを設定します。
	 *
	 * @param dhcpConfigs DHCPコンフィグ情報リスト
	 */
	public void setDhcpConfigs(List<DhcpConfig> dhcpConfigs) {
		this.dhcpConfigs = dhcpConfigs;
	}

	/**
	 * DHCPコンフィグ情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param dhcpConfigs DHCPコンフィグ情報の配列
	 * @return 自オブジェクト
	 */
	public NiftyDescribeDhcpConfigsResult withDhcpConfigs(DhcpConfig ... dhcpConfigs) {
		if (this.dhcpConfigs == null) this.dhcpConfigs = new ArrayList<DhcpConfig>();
		for (DhcpConfig dhcpConfig : dhcpConfigs) {
			getDhcpConfigs().add(dhcpConfig);
		}
		return this;
	}

	/**
	 * DHCPコンフィグ情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param dhcpConfigs DHCPコンフィグ情報リスト
	 * @return 自オブジェクト
	 */
	public NiftyDescribeDhcpConfigsResult withDhcpConfigs(Collection<DhcpConfig> dhcpConfigs) {
		if (this.dhcpConfigs == null) this.dhcpConfigs = new ArrayList<DhcpConfig>();
		if (dhcpConfigs != null) {
			getDhcpConfigs().addAll(dhcpConfigs);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[requestId=");
		builder.append(requestId);
		builder.append(", dhcpConfigs=");
		builder.append(dhcpConfigs);
		builder.append("]");
		return builder.toString();
	}
}
