/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * DHCPオプション情報クラス。<br />
 * このクラスはDHCPオプション情報を格納します。
 *
 */
public class DhcpOptions {

	/** DHCPオプションID */
	private String dhcpOptionsId;

	/** DHCPオプション設定情報 */
	private List<DhcpConfiguration> dhcpConfigurations;

	/** リソースに設定されているタグ情報 */
	private List<Tag> tagSet;

	/**
	 * DHCPオプションIDを取得します。
	 *
	 * @return DHCPオプションID
	 */
	public String getDhcpOptionsId() {
		return dhcpOptionsId;
	}

	/**
	 * DHCPオプションIDを設定します。
	 *
	 * @param dhcpOptionsId DHCPオプションID
	 */
	public void setDhcpOptionsId(String dhcpOptionsId) {
		this.dhcpOptionsId = dhcpOptionsId;
	}

	/**
	 * DHCPオプションIDを設定し、自オブジェクトを返します。
	 *
	 * @param dhcpOptionsId DHCPオプションID
	 * @return 自オブジェクト
	 */
	public DhcpOptions withDhcpOptionsId(String dhcpOptionsId) {
		setDhcpOptionsId(dhcpOptionsId);
		return this;
	}

	/**
	 * DHCPオプション設定情報を取得します。
	 *
	 * @return DHCPオプション設定情報
	 */
	public List<DhcpConfiguration> getDhcpConfigurations() {
		return dhcpConfigurations;
	}

	/**
	 * DHCPオプション設定情報を設定します。
	 *
	 * @param dhcpConfigurations DHCPオプション設定情報
	 */
	public void setDhcpConfigurations(List<DhcpConfiguration> dhcpConfigurations) {
		this.dhcpConfigurations = dhcpConfigurations;
	}

	/**
	 * DHCPオプション設定情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param dhcpConfigurations DHCPオプション設定情報の配列
	 * @return 自オブジェクト
	 */
	public DhcpOptions withDhcpConfigurations(DhcpConfiguration ... dhcpConfigurations) {
		if (this.dhcpConfigurations == null) this.dhcpConfigurations = new ArrayList<DhcpConfiguration>();
		for (DhcpConfiguration dhcpConfiguration : dhcpConfigurations) {
			getDhcpConfigurations().add(dhcpConfiguration);
		}
		return this;
	}

	/**
	 * DHCPオプション設定情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param dhcpConfigurations DHCPオプション設定情報リスト
	 * @return 自オブジェクト
	 */
	public DhcpOptions withDhcpConfigurations(Collection<DhcpConfiguration> dhcpConfigurations) {
		if (this.dhcpConfigurations == null) this.dhcpConfigurations = new ArrayList<DhcpConfiguration>();
		if (dhcpConfigurations != null) {
			getDhcpConfigurations().addAll(dhcpConfigurations);
		}
		return this;
	}

	/**
	 * リソースに設定されているタグ情報を取得します。
	 *
	 * @return リソースに設定されているタグ情報
	 */
	public List<Tag> getTagSet() {
		return tagSet;
	}

	/**
	 * リソースに設定されているタグ情報を設定します。
	 *
	 * @param tagSet リソースに設定されているタグ情報
	 */
	public void setTagSet(List<Tag> tagSet) {
		this.tagSet = tagSet;
	}

	/**
	 * リソースに設定されているタグ情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param tagSet リソースに設定されているタグ情報の配列
	 * @return 自オブジェクト
	 */
	public DhcpOptions withTagSet(Tag ... tagSet) {
		if (this.tagSet == null) this.tagSet = new ArrayList<Tag>();
		for (Tag tag : tagSet) {
			getTagSet().add(tag);
		}
		return this;
	}

	/**
	 * リソースに設定されているタグ情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param tagSet リソースに設定されているタグ情報リスト
	 * @return 自オブジェクト
	 */
	public DhcpOptions withTagSet(Collection<Tag> tagSet) {
		if (this.tagSet == null) this.tagSet = new ArrayList<Tag>();
		if (tagSet != null) {
			getTagSet().addAll(tagSet);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[dhcpOptionsId=");
		builder.append(dhcpOptionsId);
		builder.append(", dhcpConfigurations=");
		builder.append(dhcpConfigurations);
		builder.append(", tagSet=");
		builder.append(tagSet);
		builder.append("]");
		return builder.toString();
	}
}
