/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.annotation.Query;

/**
 * DHCPオプション設定情報クラス。<br />
 * このクラスはDHCPオプション設定情報を格納します。
 *
 */
public class DhcpConfiguration {

	/** DHCPオプション名 */
	private String key;

	/** DHCPオプションの値 */
	private List<String> values;

	/**
	 * DHCPオプション名を取得します。
	 *
	 * @return DHCPオプション名
	 */
	@Query(name="Key", require=true)
	public String getKey() {
		return key;
	}

	/**
	 * DHCPオプション名を設定します。
	 *
	 * @param key DHCPオプション名
	 */
	public void setKey(String key) {
		this.key = key;
	}

	/**
	 * DHCPオプション名を設定し、自オブジェクトを返します。
	 *
	 * @param key DHCPオプション名
	 * @return 自オブジェクト
	 */
	public DhcpConfiguration withKey(String key) {
		setKey(key);
		return this;
	}

	/**
	 * DHCPオプションの値を取得します。
	 *
	 * @return DHCPオプションの値
	 */
	@Query(name="Value", require=true)
	public List<String> getValues() {
		return values;
	}

	/**
	 * DHCPオプションの値を設定します。
	 *
	 * @param values DHCPオプションの値
	 */
	public void setValues(List<String> values) {
		this.values = values;
	}

	/**
	 * DHCPオプションの値の配列を設定し、自オブジェクトを返します。
	 *
	 * @param values DHCPオプションの値の配列
	 * @return 自オブジェクト
	 */
	public DhcpConfiguration withValues(String ... values) {
		if (this.values == null) this.values = new ArrayList<String>();
		for (String value : values) {
			getValues().add(value);
		}
		return this;
	}

	/**
	 * DHCPオプションの値リストを設定し、自オブジェクトを返します。
	 *
	 * @param values DHCPオプションの値リスト
	 * @return 自オブジェクト
	 */
	public DhcpConfiguration withValues(Collection<String> values) {
		if (this.values == null) this.values = new ArrayList<String>();
		if (values != null) {
			getValues().addAll(values);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[key=");
		builder.append(key);
		builder.append(", values=");
		builder.append(values);
		builder.append("]");
		return builder.toString();
	}
}
