/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * DHCPコンフィグ情報クラス。<br />
 * このクラスはDHCPコンフィグ情報を格納します。
 *
 */
public class DhcpConfig {

	/** DHCPコンフィグID */
	private String dhcpConfigId;

	/** IpAddressPool設定情報リスト */
	private List<IpAddressPool> ipAddressPools;

	/** StaticMapping設定情報リスト */
	private List<StaticMapping> staticMappings;

	/**
	 * DHCPコンフィグIDを取得します。
	 *
	 * @return DHCPコンフィグID
	 */
	public String getDhcpConfigId() {
		return dhcpConfigId;
	}

	/**
	 * DHCPコンフィグIDを設定します。
	 *
	 * @param dhcpConfigId DHCPコンフィグID
	 */
	public void setDhcpConfigId(String dhcpConfigId) {
		this.dhcpConfigId = dhcpConfigId;
	}

	/**
	 * DHCPコンフィグIDを設定し、自オブジェクトを返します。
	 *
	 * @param dhcpConfigId DHCPコンフィグID
	 * @return 自オブジェクト
	 */
	public DhcpConfig withDhcpConfigId(String dhcpConfigId) {
		setDhcpConfigId(dhcpConfigId);
		return this;
	}

	/**
	 * IpAddressPool設定情報リストを取得します。
	 *
	 * @return IpAddressPool設定情報リスト
	 */
	public List<IpAddressPool> getIpAddressPools() {
		return ipAddressPools;
	}

	/**
	 * IpAddressPool設定情報リストを設定します。
	 *
	 * @param ipAddressPools IpAddressPool設定情報リスト
	 */
	public void setIpAddressPools(List<IpAddressPool> ipAddressPools) {
		this.ipAddressPools = ipAddressPools;
	}

	/**
	 * IpAddressPool設定情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param ipAddressPools IpAddressPool設定情報の配列
	 * @return 自オブジェクト
	 */
	public DhcpConfig withIpAddressPools(IpAddressPool ... ipAddressPools) {
		if (this.ipAddressPools == null) this.ipAddressPools = new ArrayList<IpAddressPool>();
		for (IpAddressPool ipAddressPool : ipAddressPools) {
			getIpAddressPools().add(ipAddressPool);
		}
		return this;
	}

	/**
	 * IpAddressPool設定情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param ipAddressPools IpAddressPool設定情報リスト
	 * @return 自オブジェクト
	 */
	public DhcpConfig withIpAddressPools(Collection<IpAddressPool> ipAddressPools) {
		if (this.ipAddressPools == null) this.ipAddressPools = new ArrayList<IpAddressPool>();
		if (ipAddressPools != null) {
			getIpAddressPools().addAll(ipAddressPools);
		}
		return this;
	}

	/**
	 * StaticMapping設定情報リストを取得します。
	 *
	 * @return StaticMapping設定情報リスト
	 */
	public List<StaticMapping> getStaticMappings() {
		return staticMappings;
	}

	/**
	 * StaticMapping設定情報リストを設定します。
	 *
	 * @param staticMappings StaticMapping設定情報リスト
	 */
	public void setStaticMappings(List<StaticMapping> staticMappings) {
		this.staticMappings = staticMappings;
	}

	/**
	 * StaticMapping設定情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param staticMappings StaticMapping設定情報の配列
	 * @return 自オブジェクト
	 */
	public DhcpConfig withStaticMappings(StaticMapping ... staticMappings) {
		if (this.staticMappings == null) this.staticMappings = new ArrayList<StaticMapping>();
		for (StaticMapping staticMapping : staticMappings) {
			getStaticMappings().add(staticMapping);
		}
		return this;
	}

	/**
	 * StaticMapping設定情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param staticMappings StaticMapping設定情報リスト
	 * @return 自オブジェクト
	 */
	public DhcpConfig withStaticMappings(Collection<StaticMapping> staticMappings) {
		if (this.staticMappings == null) this.staticMappings = new ArrayList<StaticMapping>();
		if (staticMappings != null) {
			getStaticMappings().addAll(staticMappings);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[dhcpConfigId=");
		builder.append(dhcpConfigId);
		builder.append(", ipAddressPools=");
		builder.append(ipAddressPools);
		builder.append(", staticMappings=");
		builder.append(staticMappings);
		builder.append("]");
		return builder.toString();
	}
}
