/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Request;
import com.nifty.cloud.sdk.annotation.Action;
import com.nifty.cloud.sdk.annotation.Query;

/**
 * CreateDhcpOptionsリクエストクラス。<br />
 * このクラスはCreateDhcpOptionsへのリクエストを構築します。
 *
 */
@Action("CreateDhcpOptions")
public class CreateDhcpOptionsRequest implements Request {

	/** DHCPオプション設定情報リスト */
	private List<DhcpConfiguration> dhcpConfigurations;

	/**
	 * デフォルトコンストラクタ。
	 */
	public CreateDhcpOptionsRequest() {
	}

	/**
	 * DHCPオプション設定情報リストを指定し、CreateDhcpOptionsRequestを構築します。
	 *
	 * @param dhcpConfigurations DHCPオプション設定情報リスト
	 */
	public CreateDhcpOptionsRequest(List<DhcpConfiguration> dhcpConfigurations) {
		this.dhcpConfigurations = dhcpConfigurations;
	}

	/**
	 * DHCPオプション設定情報リストを取得します。
	 *
	 * @return DHCPオプション設定情報リスト
	 */
	@Query(name="DhcpConfiguration", require=true)
	public List<DhcpConfiguration> getDhcpConfigurations() {
		return dhcpConfigurations;
	}

	/**
	 * DHCPオプション設定情報リストを設定します。
	 *
	 * @param dhcpConfigurations DHCPオプション設定情報リスト
	 */
	public void setDhcpConfigurations(List<DhcpConfiguration> dhcpConfigurations) {
		this.dhcpConfigurations = dhcpConfigurations;
	}

	/**
	 * DHCPオプション設定情報の配列を設定し、自オブジェクトを返します。
	 *
	 * @param dhcpConfigurations DHCPオプション設定情報の配列
	 * @return 自オブジェクト
	 */
	public CreateDhcpOptionsRequest withDhcpConfigurations(DhcpConfiguration ... dhcpConfigurations) {
		if (this.dhcpConfigurations == null) this.dhcpConfigurations = new ArrayList<DhcpConfiguration>();
		for (DhcpConfiguration dhcpConfiguration : dhcpConfigurations) {
			getDhcpConfigurations().add(dhcpConfiguration);
		}
		return this;
	}

	/**
	 * DHCPオプション設定情報リストを設定し、自オブジェクトを返します。
	 *
	 * @param dhcpConfigurations DHCPオプション設定情報リスト
	 * @return 自オブジェクト
	 */
	public CreateDhcpOptionsRequest withDhcpConfigurations(Collection<DhcpConfiguration> dhcpConfigurations) {
		if (this.dhcpConfigurations == null) this.dhcpConfigurations = new ArrayList<DhcpConfiguration>();
		if (dhcpConfigurations != null) {
			getDhcpConfigurations().addAll(dhcpConfigurations);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[dhcpConfigurations=");
		builder.append(dhcpConfigurations);
		builder.append("]");
		return builder.toString();
	}
}
