/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.dhcp;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.dhcp.model.CreateDhcpOptionsRequest;
import com.nifty.cloud.sdk.dhcp.model.CreateDhcpOptionsResult;
import com.nifty.cloud.sdk.dhcp.model.DeleteDhcpOptionsRequest;
import com.nifty.cloud.sdk.dhcp.model.DeleteDhcpOptionsResult;
import com.nifty.cloud.sdk.dhcp.model.DescribeDhcpOptionsRequest;
import com.nifty.cloud.sdk.dhcp.model.DescribeDhcpOptionsResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpConfigRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpConfigResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpIpAddressPoolRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpIpAddressPoolResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpStaticMappingRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpStaticMappingResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpConfigRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpConfigResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpIpAddressPoolRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpIpAddressPoolResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpStaticMappingRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpStaticMappingResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyDescribeDhcpConfigsRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyDescribeDhcpConfigsResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyDescribeDhcpStatusRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyDescribeDhcpStatusResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyDisableDhcpRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyDisableDhcpResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyEnableDhcpRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyEnableDhcpResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyReplaceDhcpConfigRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyReplaceDhcpConfigResult;
import com.nifty.cloud.sdk.dhcp.model.NiftyReplaceDhcpOptionRequest;
import com.nifty.cloud.sdk.dhcp.model.NiftyReplaceDhcpOptionResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * DHCPカテゴリクライアント。<br />
 * DHCP カテゴリの非同期クライアントの具象クラスです。
 *
 */
public class NiftyDhcpAsyncClient extends NiftyDhcpClient implements NiftyDhcpAsync {


	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、DHCPカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftyDhcpAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報、クライアント設定を指定し、DHCPカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftyDhcpAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、DHCPカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyDhcpAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;

	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、DHCPカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyDhcpAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyEnableDhcpAsync(com.nifty.cloud.sdk.dhcp.model.NiftyEnableDhcpRequest)
	 */
	@Override
	public Future<NiftyEnableDhcpResult> niftyEnableDhcpAsync(final NiftyEnableDhcpRequest request) {
		return executorService.submit(
				new Callable<NiftyEnableDhcpResult>() {
					@Override
					public NiftyEnableDhcpResult call() {
						return niftyEnableDhcp(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyDisableDhcpAsync(com.nifty.cloud.sdk.dhcp.model.NiftyDisableDhcpRequest)
	 */
	@Override
	public Future<NiftyDisableDhcpResult> niftyDisableDhcpAsync(final NiftyDisableDhcpRequest request) {
		return executorService.submit(
				new Callable<NiftyDisableDhcpResult>() {
					@Override
					public NiftyDisableDhcpResult call() {
						return niftyDisableDhcp(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#createDhcpOptionsAsync(com.nifty.cloud.sdk.dhcp.model.CreateDhcpOptionsRequest)
	 */
	@Override
	public Future<CreateDhcpOptionsResult> createDhcpOptionsAsync(final CreateDhcpOptionsRequest request) {
		return executorService.submit(
				new Callable<CreateDhcpOptionsResult>() {
					@Override
					public CreateDhcpOptionsResult call() {
						return createDhcpOptions(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#deleteDhcpOptionsAsync(com.nifty.cloud.sdk.dhcp.model.DeleteDhcpOptionsRequest)
	 */
	@Override
	public Future<DeleteDhcpOptionsResult> deleteDhcpOptionsAsync(final DeleteDhcpOptionsRequest request) {
		return executorService.submit(
				new Callable<DeleteDhcpOptionsResult>() {
					@Override
					public DeleteDhcpOptionsResult call() {
						return deleteDhcpOptions(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#describeDhcpOptionsAsync(com.nifty.cloud.sdk.dhcp.model.DescribeDhcpOptionsRequest)
	 */
	@Override
	public Future<DescribeDhcpOptionsResult> describeDhcpOptionsAsync(final DescribeDhcpOptionsRequest request) {
		return executorService.submit(
				new Callable<DescribeDhcpOptionsResult>() {
					@Override
					public DescribeDhcpOptionsResult call() {
						return describeDhcpOptions(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyReplaceDhcpOptionAsync(com.nifty.cloud.sdk.dhcp.model.NiftyReplaceDhcpOptionRequest)
	 */
	@Override
	public Future<NiftyReplaceDhcpOptionResult> niftyReplaceDhcpOptionAsync(final NiftyReplaceDhcpOptionRequest request) {
		return executorService.submit(
				new Callable<NiftyReplaceDhcpOptionResult>() {
					@Override
					public NiftyReplaceDhcpOptionResult call() {
						return niftyReplaceDhcpOption(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyDescribeDhcpStatusAsync(com.nifty.cloud.sdk.dhcp.model.NiftyDescribeDhcpStatusRequest)
	 */
	@Override
	public Future<NiftyDescribeDhcpStatusResult> niftyDescribeDhcpStatusAsync(final NiftyDescribeDhcpStatusRequest request) {
		return executorService.submit(
				new Callable<NiftyDescribeDhcpStatusResult>() {
					@Override
					public NiftyDescribeDhcpStatusResult call() {
						return niftyDescribeDhcpStatus(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyCreateDhcpConfigAsync(com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpConfigRequest)
	 */
	@Override
	public Future<NiftyCreateDhcpConfigResult> niftyCreateDhcpConfigAsync(final NiftyCreateDhcpConfigRequest request) {
		return executorService.submit(
				new Callable<NiftyCreateDhcpConfigResult>() {
					@Override
					public NiftyCreateDhcpConfigResult call() {
						return niftyCreateDhcpConfig(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyDeleteDhcpConfigAsync(com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpConfigRequest)
	 */
	@Override
	public Future<NiftyDeleteDhcpConfigResult> niftyDeleteDhcpConfigAsync(final NiftyDeleteDhcpConfigRequest request) {
		return executorService.submit(
				new Callable<NiftyDeleteDhcpConfigResult>() {
					@Override
					public NiftyDeleteDhcpConfigResult call() {
						return niftyDeleteDhcpConfig(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyDescribeDhcpConfigsAsync(com.nifty.cloud.sdk.dhcp.model.NiftyDescribeDhcpConfigsRequest)
	 */
	@Override
	public Future<NiftyDescribeDhcpConfigsResult> niftyDescribeDhcpConfigsAsync(final NiftyDescribeDhcpConfigsRequest request) {
		return executorService.submit(
				new Callable<NiftyDescribeDhcpConfigsResult>() {
					@Override
					public NiftyDescribeDhcpConfigsResult call() {
						return niftyDescribeDhcpConfigs(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyReplaceDhcpConfigAsync(com.nifty.cloud.sdk.dhcp.model.NiftyReplaceDhcpConfigRequest)
	 */
	@Override
	public Future<NiftyReplaceDhcpConfigResult> niftyReplaceDhcpConfigAsync(final NiftyReplaceDhcpConfigRequest request) {
		return executorService.submit(
				new Callable<NiftyReplaceDhcpConfigResult>() {
					@Override
					public NiftyReplaceDhcpConfigResult call() {
						return niftyReplaceDhcpConfig(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyCreateDhcpStaticMappingAsync(com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpStaticMappingRequest)
	 */
	@Override
	public Future<NiftyCreateDhcpStaticMappingResult> niftyCreateDhcpStaticMappingAsync(final NiftyCreateDhcpStaticMappingRequest request) {
		return executorService.submit(
				new Callable<NiftyCreateDhcpStaticMappingResult>() {
					@Override
					public NiftyCreateDhcpStaticMappingResult call() {
						return niftyCreateDhcpStaticMapping(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyDeleteDhcpStaticMappingAsync(com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpStaticMappingRequest)
	 */
	@Override
	public Future<NiftyDeleteDhcpStaticMappingResult> niftyDeleteDhcpStaticMappingAsync(final NiftyDeleteDhcpStaticMappingRequest request) {
		return executorService.submit(
				new Callable<NiftyDeleteDhcpStaticMappingResult>() {
					@Override
					public NiftyDeleteDhcpStaticMappingResult call() {
						return niftyDeleteDhcpStaticMapping(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyCreateDhcpIpAddressPoolAsync(com.nifty.cloud.sdk.dhcp.model.NiftyCreateDhcpIpAddressPoolRequest)
	 */
	@Override
	public Future<NiftyCreateDhcpIpAddressPoolResult> niftyCreateDhcpIpAddressPoolAsync(final NiftyCreateDhcpIpAddressPoolRequest request) {
		return executorService.submit(
				new Callable<NiftyCreateDhcpIpAddressPoolResult>() {
					@Override
					public NiftyCreateDhcpIpAddressPoolResult call() {
						return niftyCreateDhcpIpAddressPool(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.dhcp.NiftyDhcpAsync#niftyDeleteDhcpIpAddressPoolAsync(com.nifty.cloud.sdk.dhcp.model.NiftyDeleteDhcpIpAddressPoolRequest)
	 */
	@Override
	public Future<NiftyDeleteDhcpIpAddressPoolResult> niftyDeleteDhcpIpAddressPoolAsync(final NiftyDeleteDhcpIpAddressPoolRequest request) {
		return executorService.submit(
				new Callable<NiftyDeleteDhcpIpAddressPoolResult>() {
					@Override
					public NiftyDeleteDhcpIpAddressPoolResult call() {
						return niftyDeleteDhcpIpAddressPool(request);
					}
				}
		);
	}
}