/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.customergateway;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.customergateway.model.CreateCustomerGatewayRequest;
import com.nifty.cloud.sdk.customergateway.model.CreateCustomerGatewayResult;
import com.nifty.cloud.sdk.customergateway.model.DeleteCustomerGatewayRequest;
import com.nifty.cloud.sdk.customergateway.model.DeleteCustomerGatewayResult;
import com.nifty.cloud.sdk.customergateway.model.DescribeCustomerGatewaysRequest;
import com.nifty.cloud.sdk.customergateway.model.DescribeCustomerGatewaysResult;
import com.nifty.cloud.sdk.customergateway.model.NiftyModifyCustomerGatewayAttributeRequest;
import com.nifty.cloud.sdk.customergateway.model.NiftyModifyCustomerGatewayAttributeResult;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。
 * カスタマーゲートウェイカテゴリクライアント。<br />
 * カスタマーゲートウェイ カテゴリの非同期クライアントの具象クラスです。<br />
 * 戻り値の"利用不可"パラメータは返却値がnullまたは空となります。
 */
public class NiftyCustomerGatewayAsyncClient extends NiftyCustomerGatewayClient implements NiftyCustomerGatewayAsync {


	/** 非同期タスク実行クラス. */
	protected ExecutorService executorService = null;

	/**
	 * 認証情報を指定し、カスタマーゲートウェイカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 */
	public NiftyCustomerGatewayAsyncClient(Credentials credential) {
		this(credential, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報、クライアント設定を指定し、カスタマーゲートウェイカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 */
	public NiftyCustomerGatewayAsyncClient(Credentials credential, ClientConfiguration configuration) {
		this(credential, configuration, Executors.newCachedThreadPool());
	}

	/**
	 * 認証情報,非同期タスク実行クラスを指定し、カスタマーゲートウェイカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyCustomerGatewayAsyncClient(Credentials credential, ExecutorService executorService) {
		super(credential);
		this.executorService = executorService;

	}

	/**
	 * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、カスタマーゲートウェイカテゴリクライアントを構築します。
	 *
	 * @param credential 認証情報
	 * @param configuration クライアント設定
	 * @param executorService 非同期タスク実行クラス
	 */
	public NiftyCustomerGatewayAsyncClient(Credentials credential, ClientConfiguration configuration, ExecutorService executorService) {
		super(credential, configuration);
		this.executorService = executorService;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.customergateway.NiftyCustomerGatewayAsync#createCustomerGatewayAsync(com.nifty.cloud.sdk.customergateway.model.CreateCustomerGatewayRequest)
	 */
	@Override
	public Future<CreateCustomerGatewayResult> createCustomerGatewayAsync(final CreateCustomerGatewayRequest request) {
		return executorService.submit(
				new Callable<CreateCustomerGatewayResult>() {
					@Override
					public CreateCustomerGatewayResult call() {
						return createCustomerGateway(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.customergateway.NiftyCustomerGatewayAsync#deleteCustomerGatewayAsync(com.nifty.cloud.sdk.customergateway.model.DeleteCustomerGatewayRequest)
	 */
	@Override
	public Future<DeleteCustomerGatewayResult> deleteCustomerGatewayAsync(final DeleteCustomerGatewayRequest request) {
		return executorService.submit(
				new Callable<DeleteCustomerGatewayResult>() {
					@Override
					public DeleteCustomerGatewayResult call() {
						return deleteCustomerGateway(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.customergateway.NiftyCustomerGatewayAsync#describeCustomerGatewaysAsync(com.nifty.cloud.sdk.customergateway.model.DescribeCustomerGatewaysRequest)
	 */
	@Override
	public Future<DescribeCustomerGatewaysResult> describeCustomerGatewaysAsync(final DescribeCustomerGatewaysRequest request) {
		return executorService.submit(
				new Callable<DescribeCustomerGatewaysResult>() {
					@Override
					public DescribeCustomerGatewaysResult call() {
						return describeCustomerGateways(request);
					}
				}
		);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nifty.cloud.sdk.customergateway.NiftyCustomerGatewayAsync#niftyModifyCustomerGatewayAttributeAsync(com.nifty.cloud.sdk.customergateway.model.NiftyModifyCustomerGatewayAttributeRequest)
	 */
	@Override
	public Future<NiftyModifyCustomerGatewayAttributeResult> niftyModifyCustomerGatewayAttributeAsync(final NiftyModifyCustomerGatewayAttributeRequest request) {
		return executorService.submit(
				new Callable<NiftyModifyCustomerGatewayAttributeResult>() {
					@Override
					public NiftyModifyCustomerGatewayAttributeResult call() {
						return niftyModifyCustomerGatewayAttribute(request);
					}
				}
		);
	}
}