/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.autoscale;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.auth.Credentials;
import com.nifty.cloud.sdk.autoscale.model.NiftyCreateAutoScalingGroupRequest;
import com.nifty.cloud.sdk.autoscale.model.NiftyCreateAutoScalingGroupResult;
import com.nifty.cloud.sdk.autoscale.model.NiftyDeleteAutoScalingGroupRequest;
import com.nifty.cloud.sdk.autoscale.model.NiftyDeleteAutoScalingGroupResult;
import com.nifty.cloud.sdk.autoscale.model.NiftyDescribeAutoScalingGroupsRequest;
import com.nifty.cloud.sdk.autoscale.model.NiftyDescribeAutoScalingGroupsResult;
import com.nifty.cloud.sdk.autoscale.model.NiftyDescribeScalingActivitiesRequest;
import com.nifty.cloud.sdk.autoscale.model.NiftyDescribeScalingActivitiesResult;
import com.nifty.cloud.sdk.autoscale.model.NiftyUpdateAutoScalingGroupRequest;
import com.nifty.cloud.sdk.autoscale.model.NiftyUpdateAutoScalingGroupResult;

/**
 * NIFCLOUD API クライアント 具象クラス。 オートスケール カテゴリ クライアント。<br />
 * オートスケール カテゴリの非同期クライアントの具象クラスです。
 *
 */
public class NiftyAutoScaleAsyncClient extends NiftyAutoScaleClient implements NiftyAutoScaleAsync {

    /** 非同期タスク実行クラス. */
    protected ExecutorService executorService = null;

    /**
     * 認証情報を指定し、オートスケールカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     */
    public NiftyAutoScaleAsyncClient(Credentials credential) {
        this(credential, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報、クライアント設定を指定し、オートスケールカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     */
    public NiftyAutoScaleAsyncClient(Credentials credential, ClientConfiguration configuration) {
        this(credential, configuration, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報,非同期タスク実行クラスを指定し、オートスケールカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftyAutoScaleAsyncClient(Credentials credential, ExecutorService executorService) {
        super(credential);
        this.executorService = executorService;

    }

    /**
     * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、オートスケールカテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftyAutoScaleAsyncClient(Credentials credential, ClientConfiguration configuration,
        ExecutorService executorService) {
        super(credential, configuration);
        this.executorService = executorService;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyCreateAutoScalingGroupResult>
        niftyCreateAutoCallingGroupAsync(final NiftyCreateAutoScalingGroupRequest request) {
        return executorService.submit(
            new Callable<NiftyCreateAutoScalingGroupResult>() {
                @Override
                public NiftyCreateAutoScalingGroupResult call() {
                    return niftyCreateAutoScalingGroup(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDescribeAutoScalingGroupsResult> niftyDescribeAutoScalingGroupsAsync(
        final NiftyDescribeAutoScalingGroupsRequest request) {
        return executorService.submit(
            new Callable<NiftyDescribeAutoScalingGroupsResult>() {
                @Override
                public NiftyDescribeAutoScalingGroupsResult call() {
                    return niftyDescribeAutoScalingGroups(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDescribeScalingActivitiesResult> niftyDescribeScalingActivitiesAsync(
        final NiftyDescribeScalingActivitiesRequest request) {
        return executorService.submit(
            new Callable<NiftyDescribeScalingActivitiesResult>() {
                @Override
                public NiftyDescribeScalingActivitiesResult call() {
                    return niftyDescribeScalingActivities(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyUpdateAutoScalingGroupResult>
        niftyUpdateAutoScalingGroupAsync(final NiftyUpdateAutoScalingGroupRequest request) {
        return executorService.submit(
            new Callable<NiftyUpdateAutoScalingGroupResult>() {
                @Override
                public NiftyUpdateAutoScalingGroupResult call() {
                    return niftyUpdateAutoScalingGroup(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDeleteAutoScalingGroupResult>
        niftyDeleteAutoScalingGroupAsync(final NiftyDeleteAutoScalingGroupRequest request) {
        return executorService.submit(
            new Callable<NiftyDeleteAutoScalingGroupResult>() {
                @Override
                public NiftyDeleteAutoScalingGroupResult call() {
                    return niftyDeleteAutoScalingGroup(request);
                }
            });
    }

}
