/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.alarm.model.transform;

import java.util.ArrayList;
import java.util.List;

import com.nifty.cloud.sdk.alarm.model.AlarmReservationSetDescribe;
import com.nifty.cloud.sdk.alarm.model.AlarmTargetsSet;
import com.nifty.cloud.sdk.alarm.model.EmailAddressSet;
import com.nifty.cloud.sdk.alarm.model.RuleSet;
import com.nifty.cloud.sdk.tools.DateUtils;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

/**
 * AlarmReservationSetType変換クラス。<br />
 * このクラスはレスポンスXMLのreservationSet要素からAlarmReservationSetDescribeクラスのリストを生成します。
 */
public class AlarmReservationSetDescribeConverter implements Converter {
    private static final String ELEMENT_RULE_NAME = "ruleName";
    private static final String ELEMENT_DESCRIPTION = "description";
    private static final String ELEMENT_ALARM_STATE = "alarmState";
    private static final String ELEMENT_FUNCTION_NAME = "functionName";
    private static final String ELEMENT_ZONE = "zone";
    private static final String ELEMENT_RULE_SET = "ruleSet";
    private static final String ELEMENT_ALARM_CONDITION = "alarmCondition";
    private static final String ELEMENT_EMAIL_ADDRESS_SET = "emailAddressSet";
    private static final String ELEMENT_ALARM_TARGETS_SET = "alarmTargetsSet";
    private static final String ELEMENT_CREATED_TIME = "createdTime";

    /**
     * 変換可能なクラスか判定します。
     *
     * @param clazz
     *            クラス
     * @return 変換可能な場合true, 変換不可能な場合false
     *
     * @see com.thoughtworks.xstream.converters.Converter#canConvert(Class)
     */
    @SuppressWarnings("rawtypes")
    @Override
    public boolean canConvert(Class clazz) {
        Class[] interfaces = clazz.getInterfaces();
        for (Class i : interfaces) {
            if (i == List.class) {
                return true;
            }
        }
        return false;
    }

    /**
     * {@literal
     * reservationSet要素をList<AlarmReservationSetDescribe>クラスに変換します。}
     *
     * @param reader
     *            XML読込リーダ
     * @param context
     * @return {@literal List<AlarmReservationSetDescribe>オブジェクト}
     *
     * @see com.thoughtworks.xstream.converters.Converter#unmarshal(HierarchicalStreamReader,
     *      UnmarshallingContext)
     */
    @SuppressWarnings("unchecked")
    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {

        List<AlarmReservationSetDescribe> reservations =
            new ArrayList<AlarmReservationSetDescribe>();

        Converter ruleSetConverter = new RuleSetConverter();
        Converter emailAddressSetConverter = new EmailAddressSetConverter();
        Converter alarmTargetsSetConverter = new AlarmTargetsSetConverter();

        String name = null;
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            AlarmReservationSetDescribe alarmReservationSet =
                new AlarmReservationSetDescribe();

            while (reader.hasMoreChildren()) {
                reader.moveDown();

                name = reader.getNodeName();
                if (name.equals(ELEMENT_RULE_NAME)) {
                    alarmReservationSet.setRuleName(reader.getValue());
                } else if (name.equals(ELEMENT_DESCRIPTION)) {
                    alarmReservationSet.setDescription(reader.getValue());
                } else if (name.equals(ELEMENT_ALARM_STATE)) {
                    alarmReservationSet.setAlarmState(reader.getValue());
                } else if (name.equals(ELEMENT_FUNCTION_NAME)) {
                    alarmReservationSet.setFunctionName(reader.getValue());
                } else if (name.equals(ELEMENT_ZONE)) {
                    alarmReservationSet.setZone(reader.getValue());
                } else if (name.equals(ELEMENT_RULE_SET)) {
                    alarmReservationSet
                        .setRules((List<RuleSet>) ruleSetConverter.unmarshal(reader, context));
                } else if (name.equals(ELEMENT_ALARM_CONDITION)) {
                    alarmReservationSet.setAlarmCondition(reader.getValue());
                } else if (name.equals(ELEMENT_EMAIL_ADDRESS_SET)) {
                    alarmReservationSet
                        .setEmailAddresses((List<EmailAddressSet>) emailAddressSetConverter
                            .unmarshal(reader, context));
                } else if (name.equals(ELEMENT_ALARM_TARGETS_SET)) {
                    alarmReservationSet
                        .setAlarmTargets((List<AlarmTargetsSet>) alarmTargetsSetConverter
                            .unmarshal(reader, context));
                } else if (name.equals(ELEMENT_CREATED_TIME)) {
                    alarmReservationSet.setCreatedTime(DateUtils.toDate(reader.getValue()));
                }
                reader.moveUp();
            }
            reader.moveUp();
            reservations.add(alarmReservationSet);
        }
        return reservations;
    }

    /**
     * 未実装メソッド
     */
    @Override
    public void marshal(Object paramObject,
        HierarchicalStreamWriter paramHierarchicalStreamWriter,
        MarshallingContext paramMarshallingContext) {

        // 処理無し
    }

}
