/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.alarm.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.nifty.cloud.sdk.Result;
import com.nifty.cloud.sdk.alarm.model.transform.AlarmReservationSetConverter;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * NiftyDescribeAlarmHistory結果処理クラス。<br />
 * このクラスはNiftyDescribeAlarmHistoryのレスポンスを格納します。
 */
@XStreamAlias("NiftyDescribeAlarmHistoryResponse")
public class NiftyDescribeAlarmHistoryResult extends Result {

    /** リクエスト識別子 */
    @XStreamAlias("requestId")
    private String requestId;

    /** 取得した情報セットリスト */
    @XStreamAlias("reservationSet")
    @XStreamConverter(AlarmReservationSetConverter.class)
    private List<AlarmReservationSet> reservations;

    /**
     * リクエスト識別子を取得します。
     * 
     * @return リクエスト識別子
     */
    public String getRequestId() {
        return requestId;
    }

    /**
     * リクエスト識別子を設定します。
     * 
     * @param requestId
     *            リクエスト識別子
     */
    public void setRequestId(String requestId) {
        this.requestId = requestId;
    }

    /**
     * リクエスト識別子を設定し、自オブジェクトを返します。
     * 
     * @param requestId
     *            リクエスト識別子
     * @return 自オブジェクト
     */
    public NiftyDescribeAlarmHistoryResult withRequestId(String requestId) {
        setRequestId(requestId);
        return this;
    }

    /**
     * 取得した情報セットリストを取得します。
     * 
     * @return 取得した情報セットリスト
     */
    public List<AlarmReservationSet> getReservations() {
        return reservations;
    }

    /**
     * 取得した情報セットリストを設定します。
     * 
     * @param reservations
     *            取得した情報セットリスト
     */
    public void setReservations(List<AlarmReservationSet> reservations) {
        this.reservations = reservations;
    }

    /**
     * 取得した情報セットの配列を設定し、自オブジェクトを返します。
     * 
     * @param reservations
     *            取得した情報セットの配列
     * @return 自オブジェクト
     */
    public NiftyDescribeAlarmHistoryResult withReservations(AlarmReservationSet... reservations) {
        if (this.reservations == null)
            this.reservations = new ArrayList<AlarmReservationSet>();
        for (AlarmReservationSet reservation : reservations) {
            getReservations().add(reservation);
        }
        return this;
    }

    /**
     * 取得した情報セットリストを設定し、自オブジェクトを返します。
     *
     * @param reservations
     *            取得した情報セットのリスト
     * @return 自オブジェクト
     */
    public NiftyDescribeAlarmHistoryResult
        withReservations(Collection<AlarmReservationSet> reservations) {
        if (this.reservations == null)
            this.reservations = new ArrayList<AlarmReservationSet>();
        if (reservations != null) {
            getReservations().addAll(reservations);
        }
        return this;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("[requestId=");
        builder.append(requestId);
        builder.append(", reservations=");
        builder.append(reservations);
        builder.append("]");
        return builder.toString();
    }
}
