/**
 * ******************************************************************************
 *  Copyright Fujitsu.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  You may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * *****************************************************************************
 *
 *  NIFCLOUD SDK for Java
 *
 */
package com.nifty.cloud.sdk.alarm;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.nifty.cloud.sdk.ClientConfiguration;
import com.nifty.cloud.sdk.alarm.model.NiftyCreateAlarmRequest;
import com.nifty.cloud.sdk.alarm.model.NiftyCreateAlarmResult;
import com.nifty.cloud.sdk.alarm.model.NiftyDeleteAlarmRequest;
import com.nifty.cloud.sdk.alarm.model.NiftyDeleteAlarmResult;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmHistoryRequest;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmHistoryResult;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmRulesActivitiesRequest;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmRulesActivitiesResult;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmsPartitionsRequest;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmsPartitionsResult;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmsRequest;
import com.nifty.cloud.sdk.alarm.model.NiftyDescribeAlarmsResult;
import com.nifty.cloud.sdk.alarm.model.NiftyUpdateAlarmRequest;
import com.nifty.cloud.sdk.alarm.model.NiftyUpdateAlarmResult;
import com.nifty.cloud.sdk.auth.Credentials;

/**
 * NIFCLOUD API 非同期クライアント 具象クラス。 基本監視カテゴリクライアント。<br />
 * 基本監視カテゴリの非同期クライアントの具象クラスです。<br />
 */
public class NiftyAlarmAsyncClient extends NiftyAlarmClient implements NiftyAlarmAsync {

    /** 非同期タスク実行クラス. */
    protected ExecutorService executorService = null;

    /**
     * 認証情報を指定し、基本監視カテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     */
    public NiftyAlarmAsyncClient(Credentials credential) {
        this(credential, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報、クライアント設定を指定し、基本監視カテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     */
    public NiftyAlarmAsyncClient(Credentials credential, ClientConfiguration configuration) {
        this(credential, configuration, Executors.newCachedThreadPool());
    }

    /**
     * 認証情報,非同期タスク実行クラスを指定し、基本監視カテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftyAlarmAsyncClient(Credentials credential, ExecutorService executorService) {
        super(credential);
        this.executorService = executorService;
    }

    /**
     * 認証情報、クライアント設定,非同期タスク実行クラスを指定し、基本監視カテゴリクライアントを構築します。
     *
     * @param credential
     *            認証情報
     * @param configuration
     *            クライアント設定
     * @param executorService
     *            非同期タスク実行クラス
     */
    public NiftyAlarmAsyncClient(Credentials credential, ClientConfiguration configuration,
        ExecutorService executorService) {
        super(credential, configuration);
        this.executorService = executorService;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyCreateAlarmResult>
        niftyCreateAlarmAsync(final NiftyCreateAlarmRequest request) {
        return executorService.submit(
            new Callable<NiftyCreateAlarmResult>() {
                @Override
                public NiftyCreateAlarmResult call() {
                    return niftyCreateAlarm(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDescribeAlarmsResult>
        niftyDescribeAlarmsAsync(final NiftyDescribeAlarmsRequest request) {
        return executorService.submit(
            new Callable<NiftyDescribeAlarmsResult>() {
                @Override
                public NiftyDescribeAlarmsResult call() {
                    return niftyDescribeAlarms(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyUpdateAlarmResult>
        niftyUpdateAlarmAsync(final NiftyUpdateAlarmRequest request) {
        return executorService.submit(
            new Callable<NiftyUpdateAlarmResult>() {
                @Override
                public NiftyUpdateAlarmResult call() {
                    return niftyUpdateAlarm(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDeleteAlarmResult>
        niftyDeleteAlarmAsync(final NiftyDeleteAlarmRequest request) {
        return executorService.submit(
            new Callable<NiftyDeleteAlarmResult>() {
                @Override
                public NiftyDeleteAlarmResult call() {
                    return niftyDeleteAlarm(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDescribeAlarmsPartitionsResult>
        niftyDescribeAlarmsPartitionsAsync(final NiftyDescribeAlarmsPartitionsRequest request) {
        return executorService.submit(
            new Callable<NiftyDescribeAlarmsPartitionsResult>() {
                @Override
                public NiftyDescribeAlarmsPartitionsResult call() {
                    return niftyDescribeAlarmsPartitions(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDescribeAlarmHistoryResult>
        niftyDescribeAlarmHistoryAsync(final NiftyDescribeAlarmHistoryRequest request) {
        return executorService.submit(
            new Callable<NiftyDescribeAlarmHistoryResult>() {
                @Override
                public NiftyDescribeAlarmHistoryResult call() {
                    return niftyDescribeAlarmHistory(request);
                }
            });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Future<NiftyDescribeAlarmRulesActivitiesResult>
        niftyDescribeAlarmRulesActivitiesAsync(
            final NiftyDescribeAlarmRulesActivitiesRequest request) {
        return executorService.submit(
            new Callable<NiftyDescribeAlarmRulesActivitiesResult>() {
                @Override
                public NiftyDescribeAlarmRulesActivitiesResult call() {
                    return niftyDescribeAlarmRulesActivities(request);
                }
            });
    }

}
